<?php
declare(strict_types=1);

date_default_timezone_set('UTC');

// Basic CORS/JSON defaults (safe even on Apache without headers module)
header('Content-Type: application/json; charset=utf-8');

// Sessions
$config = require __DIR__ . '/../config/config.php';
session_name($config['security']['session_name']);
if (session_status() !== PHP_SESSION_ACTIVE) {
    session_set_cookie_params([
        'lifetime' => (int)($config['security']['session_lifetime'] ?? 1209600),
        'path' => '/',
        'secure' => isset($_SERVER['HTTPS']) && ($_SERVER['HTTPS'] === 'on' || $_SERVER['HTTPS'] === 1),
        'httponly' => true,
        'samesite' => 'Lax',
    ]);
    session_start();
}

function base_url(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $dir = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '')), '/.');
    if ($dir && $dir !== '/') $dir .= '/';
    return "{$scheme}://{$host}{$dir}";
}

function pdo(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;
    $cfg = require __DIR__ . '/../config/config.php';
    $dsn = "mysql:host={$cfg['db']['host']};dbname={$cfg['db']['name']};charset={$cfg['db']['charset']};port={$cfg['db']['port']}";
    try {
        $pdo = new PDO($dsn, $cfg['db']['user'], $cfg['db']['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        // ensure modern SQL mode (but don't break host if restricted)
        @$pdo->exec("SET sql_mode = ''; SET time_zone = '+00:00'");
    } catch (Throwable $e) {
        http_response_code(500);
        echo json_encode(['ok' => false, 'error' => 'DB_CONNECT_FAILED', 'message' => $e->getMessage()]);
        exit;
    }
    return $pdo;
}

function table_exists(string $table): bool {
    try {
        $stmt = pdo()->query("SHOW TABLES LIKE " . pdo()->quote($table));
        return (bool) $stmt->fetchColumn();
    } catch (Throwable $e) {
        return false;
    }
}

function get_user(): ?array {
    if (!empty($_SESSION['user'])) return $_SESSION['user'];
    return null;
}
function require_user(): array {
    $u = get_user();
    if (!$u) {
        http_response_code(401);
        echo json_encode(['ok' => false, 'error' => 'AUTH_REQUIRED']);
        exit;
    }
    return $u;
}

function param(string $key, $default=null) {
    if (isset($_POST[$key])) return $_POST[$key];
    if (isset($_GET[$key])) return $_GET[$key];
    return $default;
}

function json_ok($data=[], int $code=200) {
    http_response_code($code);
    echo json_encode(['ok' => true] + $data);
    exit;
}
function json_fail(string $code, string $message='', int $http=400, $extra=[]) {
    http_response_code($http);
    $payload = ['ok'=>false,'error'=>$code];
    if ($message !== '') $payload['message'] = $message;
    if (!empty($extra)) $payload += $extra;
    echo json_encode($payload);
    exit;
}

function boolval_str($v): bool {
    if (is_bool($v)) return $v;
    $v = strtolower(trim((string)$v));
    return in_array($v, ['1','true','yes','y','on'], true);
}

function hash_url(string $url): string {
    return sha1($url);
}

function sanitize_tags($tags): array {
    if (is_array($tags)) return array_values(array_filter(array_map('trim', $tags)));
    $tags = str_replace([';', '|'], ',', (string)$tags);
    $parts = array_filter(array_map('trim', explode(',', $tags)));
    return array_values($parts);
}

/** Compose WHERE clause fragments safely */
function build_where(array $fragments): string {
    $fragments = array_values(array_filter(array_map('trim', $fragments)));
    if (!$fragments) return '1=1';
    return implode(' AND ', $fragments);
}

function like_param(string $s): string {
    return '%' . str_replace(['%', '_'], ['\\%', '\\_'], $s) . '%';
}

function normalize_order(string $s, array $allowed, string $default): string {
    $s = strtolower(trim($s));
    return in_array($s, $allowed, true) ? $s : $default;
}

function as_int($v, int $default): int {
    $n = filter_var($v, FILTER_VALIDATE_INT);
    if ($n === false || $n === null) return $default;
    return (int)$n;
}

function clamp(int $v, int $min, int $max): int {
    if ($v < $min) return $min;
    if ($v > $max) return $max;
    return $v;
}

function format_feed_row(array $r): array {
    // Normalize a feed row for API output
    return [
        'id'            => (int)$r['id'],
        'title'         => $r['title'],
        'link'          => $r['link'],
        'thumbnail'     => $r['thumbnail'],
        'description'   => $r['description'],
        'pubDate'       => $r['pubDate'],
        'author'        => $r['author'],
        'source_domain' => $r['source_domain'],
        'likes'         => isset($r['likes']) ? (int)$r['likes'] : 0,
        'dislikes'      => isset($r['dislikes']) ? (int)$r['dislikes'] : 0,
        'views_1d'      => isset($r['views_1d']) ? (int)$r['views_1d'] : 0,
        'views_7d'      => isset($r['views_7d']) ? (int)$r['views_7d'] : 0,
        'views_14d'     => isset($r['views_14d']) ? (int)$r['views_14d'] : 0,
        'trending_score'=> isset($r['trending_score']) ? (float)$r['trending_score'] : null,
    ];
}

// Simple router helper
function route_action(): string {
    $a = param('action', '');
    return preg_replace('/[^a-z0-9_\-\.]/i', '', $a);
}
