<?php
/**
 * ============================================
 * FLOWBOT DCI v3.1 - PROCESSING VIEW
 * ============================================
 * Stage 3: Real-time processing with live updates
 * v3.1: Immediate page load + detailed stats
 */
declare(strict_types=1);

// Variables from controller (with safe defaults)
$baseUrl = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
$pageTitle = 'Processing';
$currentPage = 'process';
$csrfToken = $csrfToken ?? '';
$processId = $processId ?? '';

// v3.1: Check if this is initializing state
$isInitializing = $isInitializing ?? false;

// Data with safe defaults
$data = $data ?? [];
$logs = $logs ?? [];

// Process stats
$totalLinks = max($data['total_links'] ?? 0, 1);
$processedLinks = $data['processed_links'] ?? 0;
$importedLinks = $data['imported_links'] ?? 0;
$ignoredLinks = $data['ignored_links'] ?? 0;
$errorLinks = $data['error_links'] ?? 0;

// Calculate percentages
$processedPercent = round(($processedLinks / $totalLinks) * 100, 1);
$importedPercent = round(($importedLinks / $totalLinks) * 100, 1);
$ignoredPercent = round(($ignoredLinks / $totalLinks) * 100, 1);
$errorPercent = round(($errorLinks / $totalLinks) * 100, 1);
$totalProcessedPercent = round((($processedLinks + $ignoredLinks + $errorLinks) / $totalLinks) * 100, 1);

// Times
$elapsedTime = gmdate('H:i:s', (int)($data['elapsed_time'] ?? 0));
$remainingTime = gmdate('H:i:s', (int)($data['remaining_time'] ?? 0));
$processingRate = $data['processing_rate'] ?? 0;

// Remaining links
$remainingLinks = 0;
foreach ($data['phase_queues'] ?? [] as $queue) {
    $remainingLinks += count($queue);
}

// Current phase
$currentPhase = $data['phase_index'] ?? 0;

// Status
$isComplete = ($remainingLinks === 0 && !$isInitializing && $totalLinks > 0);
$processStatus = $data['status'] ?? ($isComplete ? 'completed' : ($isInitializing ? 'initializing' : 'running'));
$isPaused = $processStatus === 'paused';

// Domain stats
$domainStats = $data['domain_stats'] ?? [];
$totalDomains = count($domainStats);

// Success rate
$successRate = $totalLinks > 0 ? round(($importedLinks / $totalLinks) * 100, 1) : 0;

// HTTP codes
$httpCodes = $data['http_codes'] ?? [];
arsort($httpCodes);

// Current URL
$currentUrl = $data['current_url'] ?? '';

// v3.1: Detailed stats from backend
$ignoredDetails = $data['ignored_details'] ?? ['duplicate' => 0, 'invalid_url' => 0, 'blocked_domain' => 0, 'non_html' => 0];
$errorDetails = $data['error_details'] ?? ['timeout' => 0, 'http_429' => 0, 'http_404' => 0, 'http_403' => 0, 'http_5xx' => 0, 'connection' => 0, 'metadata' => 0, 'other' => 0];
$requestStats = $data['request_stats'] ?? ['total_requests' => 0, 'total_response_time' => 0, 'response_times' => []];
$retryStats = $data['retry_stats'] ?? ['total_retries' => 0, 'retries_by_phase' => [0=>0,1=>0,2=>0,3=>0], 'urls_with_retries' => 0];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - Flowb0t DCI v3.1</title>
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 32 32'><rect width='32' height='32' rx='8' fill='%233B82F6'/><path d='M8 10h16M8 16h12M8 22h8' stroke='white' stroke-width='2.5' stroke-linecap='round'/></svg>">
    <link rel="stylesheet" href="<?= $baseUrl ?>/assets/css/v3-styles.css?v=<?= time() ?>">
</head>
<body>
    <?php if ($isInitializing): ?>
    <!-- v3.1: Initializing Overlay - Shows immediately while process is being set up -->
    <div id="initializingOverlay" class="initializing-overlay">
        <div class="init-content">
            <div class="init-spinner"></div>
            <h2 class="init-title">Preparing processing...</h2>
            <div class="init-steps">
                <div class="init-step active" id="stepExtract">
                    <span class="step-icon">
                        <svg class="spinner-svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="18" height="18">
                            <circle cx="12" cy="12" r="10" stroke-opacity="0.25"/>
                            <path d="M12 2a10 10 0 0 1 10 10" stroke-linecap="round"/>
                        </svg>
                    </span>
                    <span class="step-text">Extracting URLs...</span>
                </div>
                <div class="init-step" id="stepInit">
                    <span class="step-icon">○</span>
                    <span class="step-text">Initializing tracker...</span>
                </div>
                <div class="init-step" id="stepCreate">
                    <span class="step-icon">○</span>
                    <span class="step-text">Creating process...</span>
                </div>
            </div>
            <p class="init-subtitle">Page loaded instantly! Initialization in progress...</p>
        </div>
    </div>
    <?php endif; ?>

    <!-- Navigation -->
    <nav class="main-nav">
        <div class="nav-container">
            <div class="nav-brand">
                <a href="<?= $baseUrl ?>/" class="brand-link">
                    <span class="brand-logo">
                        <svg viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <rect width="32" height="32" rx="8" fill="url(#brand-gradient)"/>
                            <path d="M8 10h16M8 16h12M8 22h8" stroke="white" stroke-width="2.5" stroke-linecap="round"/>
                            <circle cx="24" cy="22" r="4" fill="#10B981"/>
                            <defs>
                                <linearGradient id="brand-gradient" x1="0" y1="0" x2="32" y2="32">
                                    <stop stop-color="#3B82F6"/>
                                    <stop offset="1" stop-color="#8B5CF6"/>
                                </linearGradient>
                            </defs>
                        </svg>
                    </span>
                    <span class="brand-text">Flowb0t DCI</span>
                    <span class="brand-version">v3.1</span>
                </a>
            </div>

            <div class="nav-main">
                <a href="<?= $baseUrl ?>/" class="nav-link">
                    <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <rect x="3" y="3" width="7" height="7" rx="1"/>
                        <rect x="14" y="3" width="7" height="7" rx="1"/>
                        <rect x="3" y="14" width="7" height="7" rx="1"/>
                        <rect x="14" y="14" width="7" height="7" rx="1"/>
                    </svg>
                    <span>Dashboard</span>
                </a>
                <a href="/flowb0t.com/v2/views/crawler-ultimate.php" class="nav-link">
                    <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="11" cy="11" r="8"/>
                        <line x1="21" y1="21" x2="16.65" y2="16.65"/>
                    </svg>
                    <span>Crawler</span>
                </a>
                <a href="<?= $baseUrl ?>/history" class="nav-link">
                    <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="9"/>
                        <polyline points="12,7 12,12 16,14"/>
                    </svg>
                    <span>History</span>
                </a>
            </div>

            <div class="nav-right">
                <span class="status-indicator <?= $processStatus ?>" id="statusIndicator">
                    <span class="status-dot"></span>
                    <span id="statusText"><?= ucfirst($processStatus) ?></span>
                </span>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <main class="main-content" id="mainContent" style="<?= $isInitializing ? 'opacity: 0.3; pointer-events: none;' : '' ?>">
        <!-- Step Indicator -->
        <div class="step-indicator">
            <div class="step completed">
                <div class="step-circle">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
                        <polyline points="20 6 9 17 4 12"/>
                    </svg>
                </div>
                <span class="step-label">URL Input</span>
            </div>
            <div class="step-line completed"></div>
            <div class="step completed">
                <div class="step-circle">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
                        <polyline points="20 6 9 17 4 12"/>
                    </svg>
                </div>
                <span class="step-label">Analysis</span>
            </div>
            <div class="step-line <?= $isComplete ? 'completed' : '' ?>"></div>
            <div class="step <?= $isComplete ? 'completed' : 'active' ?>">
                <div class="step-circle">
                    <?php if ($isComplete): ?>
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3">
                        <polyline points="20 6 9 17 4 12"/>
                    </svg>
                    <?php else: ?>
                    3
                    <?php endif; ?>
                </div>
                <span class="step-label">Processing</span>
            </div>
        </div>

        <?php if ($isComplete): ?>
        <!-- Completion Banner -->
        <div class="completion-banner">
            <div class="completion-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="32" height="32">
                    <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                    <polyline points="22 4 12 14.01 9 11.01"/>
                </svg>
            </div>
            <div class="completion-info">
                <h2 class="completion-title">Processing Complete!</h2>
                <p class="completion-subtitle">
                    Processed <?= number_format($totalLinks) ?> URLs in <?= $elapsedTime ?>
                    with <?= $successRate ?>% success rate
                </p>
            </div>
            <div class="completion-actions">
                <a href="<?= $baseUrl ?>/?export=<?= urlencode($processId) ?>" class="btn btn-primary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                        <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                        <polyline points="7 10 12 15 17 10"/>
                        <line x1="12" y1="15" x2="12" y2="3"/>
                    </svg>
                    Export Results
                </a>
                <a href="<?= $baseUrl ?>/new" class="btn btn-secondary">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                        <circle cx="12" cy="12" r="10"/>
                        <line x1="12" y1="8" x2="12" y2="16"/>
                        <line x1="8" y1="12" x2="16" y2="12"/>
                    </svg>
                    New Process
                </a>
            </div>
        </div>
        <?php endif; ?>

        <div class="progress-layout">
            <!-- Main Progress Area -->
            <div class="progress-main">
                <!-- Progress Hero -->
                <div class="progress-hero" id="progressHero">
                    <div class="progress-circle-large">
                        <svg viewBox="0 0 100 100">
                            <circle class="progress-circle-bg" cx="50" cy="50" r="44"/>
                            <circle class="progress-circle-fill <?= $isPaused ? 'paused' : ($isComplete ? 'complete' : '') ?>"
                                    cx="50" cy="50" r="44"
                                    stroke-dasharray="276.46"
                                    stroke-dashoffset="<?= 276.46 - (276.46 * $totalProcessedPercent / 100) ?>"
                                    id="progressCircle"/>
                        </svg>
                        <div class="progress-circle-text">
                            <div class="progress-circle-value" id="progressValue"><?= $totalProcessedPercent ?>%</div>
                            <div class="progress-circle-label">Complete</div>
                        </div>
                    </div>

                    <div class="progress-hero-stats">
                        <div class="hero-stat success">
                            <div class="hero-stat-value" id="statImported"><?= number_format($importedLinks) ?></div>
                            <div class="hero-stat-label">Imported</div>
                        </div>
                        <div class="hero-stat warning">
                            <div class="hero-stat-value" id="statIgnored"><?= number_format($ignoredLinks) ?></div>
                            <div class="hero-stat-label">Ignored</div>
                        </div>
                        <div class="hero-stat error">
                            <div class="hero-stat-value" id="statErrors"><?= number_format($errorLinks) ?></div>
                            <div class="hero-stat-label">Errors</div>
                        </div>
                        <div class="hero-stat">
                            <div class="hero-stat-value" id="statRemaining"><?= number_format($remainingLinks) ?></div>
                            <div class="hero-stat-label">Remaining</div>
                        </div>
                    </div>
                </div>

                <!-- Performance ao Vivo -->
                <div class="performance-live-section" id="performanceLive">
                    <div class="performance-grid">
                        <div class="perf-stat">
                            <div class="perf-stat-header">
                                <span class="perf-stat-label">URLs/segundo</span>
                                <span class="perf-stat-trend" id="speedTrend">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="12" height="12">
                                        <polyline points="23 6 13.5 15.5 8.5 10.5 1 18"/>
                                        <polyline points="17 6 23 6 23 12"/>
                                    </svg>
                                </span>
                            </div>
                            <div class="perf-stat-value" id="perfUrlsPerSecond">0.0</div>
                            <div class="perf-stat-bar">
                                <div class="perf-stat-bar-fill" id="perfSpeedBar" style="width: 0%"></div>
                            </div>
                        </div>
                        <div class="perf-stat">
                            <div class="perf-stat-header">
                                <span class="perf-stat-label">Current Batch</span>
                            </div>
                            <div class="perf-stat-value"><span id="perfBatchCurrent">0</span>/<span id="perfBatchTotal">0</span></div>
                            <div class="perf-stat-bar">
                                <div class="perf-stat-bar-fill batch" id="perfBatchBar" style="width: 0%"></div>
                            </div>
                        </div>
                        <div class="perf-stat">
                            <div class="perf-stat-header">
                                <span class="perf-stat-label">Average Latency</span>
                            </div>
                            <div class="perf-stat-value"><span id="perfLatency">0</span>ms</div>
                            <div class="perf-stat-bar">
                                <div class="perf-stat-bar-fill latency" id="perfLatencyBar" style="width: 0%"></div>
                            </div>
                        </div>
                        <div class="perf-stat">
                            <div class="perf-stat-header">
                                <span class="perf-stat-label">Success Rate</span>
                                <span class="perf-stat-percent" id="successRateChange">+0%</span>
                            </div>
                            <div class="perf-stat-value"><span id="perfSuccessRate"><?= $successRate ?></span>%</div>
                            <div class="perf-stat-bar">
                                <div class="perf-stat-bar-fill success" id="perfSuccessBar" style="width: <?= $successRate ?>%"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Phase Pipeline -->
                <div class="phase-pipeline" id="phasePipeline">
                    <?php
                    $phaseConfigs = [
                        0 => ['concurrency' => 300, 'timeout' => 5],
                        1 => ['concurrency' => 200, 'timeout' => 8],
                        2 => ['concurrency' => 150, 'timeout' => 12],
                        3 => ['concurrency' => 100, 'timeout' => 20],
                    ];
                    for ($i = 0; $i <= 3; $i++):
                        $stats = $data['phase_stats'][$i] ?? ['success' => 0, 'retry' => 0, 'error' => 0];
                        $queueCount = count($data['phase_queues'][$i] ?? []);
                        $isActive = ($i === $currentPhase) && !$isComplete;
                        $isCompleted = ($i < $currentPhase) || $isComplete;
                        $phaseConfig = $phaseConfigs[$i];
                    ?>
                    <?php if ($i > 0): ?>
                    <div class="phase-connector <?= $isCompleted ? 'completed' : '' ?>"></div>
                    <?php endif; ?>
                    <div class="phase-node <?= $isActive ? 'active' : '' ?> <?= $isCompleted ? 'completed' : '' ?>" data-phase="<?= $i ?>">
                        <div class="phase-circle">
                            <?php if ($isCompleted && !$isActive): ?>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" width="18" height="18">
                                <polyline points="20 6 9 17 4 12"/>
                            </svg>
                            <?php else: ?>
                            <?= $i ?>
                            <?php endif; ?>
                        </div>
                        <span class="phase-label">Phase <?= $i ?></span>
                        <div class="phase-details <?= $isActive ? 'expanded' : '' ?>" id="phaseDetails<?= $i ?>">
                            <div class="phase-detail-row">
                                <span class="phase-detail-icon">⚡</span>
                                <span class="phase-detail-label">Concurrency:</span>
                                <span class="phase-detail-value"><?= $phaseConfig['concurrency'] ?></span>
                            </div>
                            <div class="phase-detail-row">
                                <span class="phase-detail-icon">⏱</span>
                                <span class="phase-detail-label">Timeout:</span>
                                <span class="phase-detail-value"><?= $phaseConfig['timeout'] ?>s</span>
                            </div>
                            <div class="phase-detail-row">
                                <span class="phase-detail-icon">✓</span>
                                <span class="phase-detail-label">Processed:</span>
                                <span class="phase-detail-value text-success" id="phaseProcessed<?= $i ?>"><?= $stats['success'] ?></span>
                            </div>
                            <div class="phase-detail-row">
                                <span class="phase-detail-icon">📋</span>
                                <span class="phase-detail-label">In queue:</span>
                                <span class="phase-detail-value" id="phaseQueue<?= $i ?>"><?= $queueCount ?></span>
                            </div>
                            <div class="phase-detail-row">
                                <span class="phase-detail-icon">🚀</span>
                                <span class="phase-detail-label">Speed:</span>
                                <span class="phase-detail-value" id="phaseSpeed<?= $i ?>">-/s</span>
                            </div>
                        </div>
                        <span class="phase-stats-mini">
                            <span class="text-success"><?= $stats['success'] ?></span> /
                            <span class="text-error"><?= $stats['error'] ?></span>
                        </span>
                    </div>
                    <?php endfor; ?>
                </div>

                <!-- v3.1: Detailed Stats Section -->
                <div class="detailed-stats-section">
                    <div class="stats-grid-detailed">
                        <!-- Ignored by Type -->
                        <div class="stat-card-detailed">
                            <h3 class="stat-card-title">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="4.93" y1="4.93" x2="19.07" y2="19.07"/>
                                </svg>
                                Ignored by Type
                            </h3>
                            <div class="stat-breakdown" id="ignoredBreakdown">
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Duplicates</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill blue" id="ignoredDuplicate" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="ignoredDuplicateVal"><?= $ignoredDetails['duplicate'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Invalid URL</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill orange" id="ignoredInvalid" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="ignoredInvalidVal"><?= $ignoredDetails['invalid_url'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Blocked Domain</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill red" id="ignoredBlocked" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="ignoredBlockedVal"><?= $ignoredDetails['blocked_domain'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Non-HTML</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill gray" id="ignoredNonHtml" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="ignoredNonHtmlVal"><?= $ignoredDetails['non_html'] ?? 0 ?></span>
                                </div>
                            </div>
                        </div>

                        <!-- Errors by Type -->
                        <div class="stat-card-detailed">
                            <h3 class="stat-card-title">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                </svg>
                                Errors by Type
                            </h3>
                            <div class="stat-breakdown" id="errorBreakdown">
                                <div class="breakdown-item">
                                    <span class="breakdown-label">HTTP 403</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill red" id="error403" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="error403Val"><?= $errorDetails['http_403'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">HTTP 404</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill orange" id="error404" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="error404Val"><?= $errorDetails['http_404'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">HTTP 429</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill yellow" id="error429" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="error429Val"><?= $errorDetails['http_429'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">HTTP 5xx</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill purple" id="error5xx" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="error5xxVal"><?= $errorDetails['http_5xx'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Timeout</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill gray" id="errorTimeout" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="errorTimeoutVal"><?= $errorDetails['timeout'] ?? 0 ?></span>
                                </div>
                                <div class="breakdown-item">
                                    <span class="breakdown-label">Connection</span>
                                    <div class="breakdown-bar"><div class="breakdown-fill brown" id="errorConnection" style="width: 0%"></div></div>
                                    <span class="breakdown-value" id="errorConnectionVal"><?= $errorDetails['connection'] ?? 0 ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Current URL (if processing) -->
                <?php if (!$isComplete && !empty($currentUrl)): ?>
                <div class="current-url-display" id="currentUrlDisplay">
                    <span class="spinner-icon">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                            <line x1="12" y1="2" x2="12" y2="6"/>
                            <line x1="12" y1="18" x2="12" y2="22"/>
                            <line x1="4.93" y1="4.93" x2="7.76" y2="7.76"/>
                            <line x1="16.24" y1="16.24" x2="19.07" y2="19.07"/>
                            <line x1="2" y1="12" x2="6" y2="12"/>
                            <line x1="18" y1="12" x2="22" y2="12"/>
                            <line x1="4.93" y1="19.07" x2="7.76" y2="16.24"/>
                            <line x1="16.24" y1="7.76" x2="19.07" y2="4.93"/>
                        </svg>
                    </span>
                    <span class="url" id="currentUrl"><?= htmlspecialchars($currentUrl) ?></span>
                </div>
                <?php endif; ?>

                <!-- v3.1: Enhanced Live Log with Filters -->
                <div class="section live-log-section">
                    <div class="section-header log-header-enhanced">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                <polyline points="14 2 14 8 20 8"/>
                            </svg>
                            Detailed Logs
                        </h2>
                        <div class="log-filters">
                            <button class="filter-btn active" data-filter="all">All</button>
                            <button class="filter-btn" data-filter="success">✓ Success</button>
                            <button class="filter-btn" data-filter="error">✗ Error</button>
                            <button class="filter-btn" data-filter="ignored">○ Ignored</button>
                        </div>
                        <input type="text" class="log-search" id="logSearch" placeholder="Filter URL or domain...">
                        <span class="badge badge-secondary" id="logCount"><?= count($logs) ?> entries</span>
                    </div>
                    <div class="section-body">
                        <div class="live-log-enhanced" id="liveLog">
                            <?php foreach (array_slice($logs, -50) as $log): ?>
                            <div class="log-entry-enhanced log-<?= $log['class'] ?? 'info' ?>" data-type="<?= $log['class'] ?? 'info' ?>">
                                <div class="log-row-main">
                                    <span class="log-timestamp"><?= isset($log['timestamp']) ? date('H:i:s', strtotime($log['timestamp'])) : '--:--:--' ?></span>
                                    <span class="log-icon">
                                        <?php if (($log['class'] ?? '') === 'success'): ?>✓
                                        <?php elseif (($log['class'] ?? '') === 'error'): ?>✗
                                        <?php else: ?>○<?php endif; ?>
                                    </span>
                                    <span class="log-code <?= (($log['http_code'] ?? 0) >= 200 && ($log['http_code'] ?? 0) < 300) ? 'success' : (($log['http_code'] ?? 0) >= 400 ? 'error' : 'info') ?>"><?= $log['http_code'] ?? '-' ?></span>
                                    <span class="log-duration"><?= isset($log['response_time']) ? round($log['response_time'] * 1000) . 'ms' : '-' ?></span>
                                    <span class="log-domain"><?= htmlspecialchars($log['domain'] ?? '-') ?></span>
                                </div>
                                <div class="log-row-url"><?= htmlspecialchars($log['url'] ?? '') ?></div>
                                <?php if (!empty($log['message'])): ?>
                                <div class="log-row-message"><?= htmlspecialchars($log['message']) ?></div>
                                <?php endif; ?>
                            </div>
                            <?php endforeach; ?>
                            <?php if (empty($logs)): ?>
                            <div class="empty-state">
                                <p class="empty-state-text">Waiting for logs...</p>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="progress-sidebar">
                <!-- Control Buttons -->
                <?php if (!$isComplete): ?>
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">Controls</h2>
                    </div>
                    <div class="section-body">
                        <div class="control-buttons" id="controlButtons">
                            <?php if ($isPaused): ?>
                            <button class="btn-control btn-resume" id="btnResume" data-id="<?= htmlspecialchars($processId) ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                    <polygon points="5 3 19 12 5 21 5 3"/>
                                </svg>
                                Resume
                            </button>
                            <?php else: ?>
                            <button class="btn-control btn-pause" id="btnPause" data-id="<?= htmlspecialchars($processId) ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                    <rect x="6" y="4" width="4" height="16"/>
                                    <rect x="14" y="4" width="4" height="16"/>
                                </svg>
                                Pause
                            </button>
                            <?php endif; ?>
                            <button class="btn-control btn-cancel" id="btnCancel" data-id="<?= htmlspecialchars($processId) ?>">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                    <circle cx="12" cy="12" r="10"/>
                                    <line x1="15" y1="9" x2="9" y2="15"/>
                                    <line x1="9" y1="9" x2="15" y2="15"/>
                                </svg>
                                Cancel
                            </button>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Time Stats -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <polyline points="12 6 12 12 16 14"/>
                            </svg>
                            Time
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="time-stats-grid">
                            <div class="time-stat">
                                <div class="time-stat-label">Elapsed</div>
                                <div class="time-stat-value" id="timeElapsed"><?= $elapsedTime ?></div>
                            </div>
                            <div class="time-stat">
                                <div class="time-stat-label">Remaining</div>
                                <div class="time-stat-value" id="timeRemaining"><?= $remainingTime ?></div>
                            </div>
                            <div class="time-stat">
                                <div class="time-stat-label">Rate</div>
                                <div class="time-stat-value" id="processingRate"><?= $processingRate ?>/s</div>
                            </div>
                            <div class="time-stat">
                                <div class="time-stat-label">Domains</div>
                                <div class="time-stat-value" id="totalDomains"><?= $totalDomains ?></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- v3.1: Request Stats with Percentiles -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M21.21 15.89A10 10 0 1 1 8 2.83"/>
                                <path d="M22 12A10 10 0 0 0 12 2v10z"/>
                            </svg>
                            Request Stats
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="request-stats-grid">
                            <div class="req-stat">
                                <span class="req-stat-label">Total Requests</span>
                                <span class="req-stat-value" id="totalRequests"><?= $requestStats['total_requests'] ?? 0 ?></span>
                            </div>
                            <div class="req-stat">
                                <span class="req-stat-label">Total Time</span>
                                <span class="req-stat-value" id="totalResponseTime"><?= round(($requestStats['total_response_time'] ?? 0), 1) ?>s</span>
                            </div>
                        </div>
                        <div class="percentiles-grid">
                            <div class="percentile">
                                <span class="percentile-label">p50</span>
                                <span class="percentile-value" id="p50">-</span>
                            </div>
                            <div class="percentile">
                                <span class="percentile-label">p90</span>
                                <span class="percentile-value" id="p90">-</span>
                            </div>
                            <div class="percentile">
                                <span class="percentile-label">p99</span>
                                <span class="percentile-value" id="p99">-</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Circuit Breaker Status -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M13 2L3 14h9l-1 8 10-12h-9l1-8z"/>
                            </svg>
                            Circuit Breaker
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="circuit-breaker-status" id="circuitBreakerStatus">
                            <div class="cb-main-status">
                                <span class="cb-indicator cb-closed" id="cbIndicator"></span>
                                <span class="cb-status-text" id="cbStatusText">Closed (OK)</span>
                            </div>
                            <div class="cb-stats">
                                <div class="cb-stat">
                                    <span class="cb-stat-label">Blocked Domains</span>
                                    <span class="cb-stat-value" id="cbBlockedDomains">0</span>
                                </div>
                                <div class="cb-stat">
                                    <span class="cb-stat-label">Remaining Cooldown</span>
                                    <span class="cb-stat-value" id="cbCooldown">-</span>
                                </div>
                            </div>
                            <div class="cb-blocked-list" id="cbBlockedList" style="display: none;">
                                <span class="cb-blocked-title">Domains in cooldown:</span>
                                <ul id="cbBlockedDomainsList"></ul>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Retry Stats -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="23 4 23 10 17 10"/>
                                <polyline points="1 20 1 14 7 14"/>
                                <path d="M3.51 9a9 9 0 0 1 14.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0 0 20.49 15"/>
                            </svg>
                            Retries
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="retry-stats" id="retryStats">
                            <div class="retry-main">
                                <div class="retry-stat-big">
                                    <span class="retry-stat-value" id="retryTotal"><?= $retryStats['total_retries'] ?? 0 ?></span>
                                    <span class="retry-stat-label">Total retries</span>
                                </div>
                                <div class="retry-stat-big success">
                                    <span class="retry-stat-value" id="retrySuccessRate">0%</span>
                                    <span class="retry-stat-label">Success Rate</span>
                                </div>
                            </div>
                            <div class="retry-by-phase">
                                <span class="retry-phase-title">Retries by phase:</span>
                                <div class="retry-phase-bars">
                                    <?php for ($i = 0; $i <= 3; $i++): ?>
                                    <div class="retry-phase-item">
                                        <span class="retry-phase-label">P<?= $i ?></span>
                                        <div class="retry-phase-bar"><div class="retry-phase-bar-fill" id="retryPhase<?= $i ?>Bar" style="width: 0%"></div></div>
                                        <span class="retry-phase-count" id="retryPhase<?= $i ?>"><?= $retryStats['retries_by_phase'][$i] ?? 0 ?></span>
                                    </div>
                                    <?php endfor; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Speed Chart -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <polyline points="22 12 18 12 15 21 9 3 6 12 2 12"/>
                            </svg>
                            Speed History
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="speed-chart-container">
                            <canvas id="speedChart" height="100"></canvas>
                        </div>
                        <div class="speed-stats-mini">
                            <div class="speed-stat">
                                <span class="speed-label">Avg</span>
                                <span class="speed-value" id="avgSpeed">-</span>
                            </div>
                            <div class="speed-stat">
                                <span class="speed-label">Peak</span>
                                <span class="speed-value" id="peakSpeed">-</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- v3.1: HTTP Codes by Category -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <path d="M2 12h20"/>
                            </svg>
                            HTTP Codes
                        </h2>
                    </div>
                    <div class="section-body">
                        <div class="http-codes-categorized" id="httpCodesCategorized">
                            <div class="http-category success">
                                <span class="http-category-label">2xx Success</span>
                                <div class="http-codes-list" id="http2xx">
                                    <?php foreach ($httpCodes as $code => $count): ?>
                                        <?php if ((int)$code >= 200 && (int)$code < 300): ?>
                                        <div class="http-code-item">
                                            <span class="http-code-badge success"><?= $code ?></span>
                                            <span class="http-code-count"><?= number_format($count) ?></span>
                                        </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="http-category redirect">
                                <span class="http-category-label">3xx Redirect</span>
                                <div class="http-codes-list" id="http3xx">
                                    <?php foreach ($httpCodes as $code => $count): ?>
                                        <?php if ((int)$code >= 300 && (int)$code < 400): ?>
                                        <div class="http-code-item">
                                            <span class="http-code-badge redirect"><?= $code ?></span>
                                            <span class="http-code-count"><?= number_format($count) ?></span>
                                        </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="http-category client-error">
                                <span class="http-category-label">4xx Client</span>
                                <div class="http-codes-list" id="http4xx">
                                    <?php foreach ($httpCodes as $code => $count): ?>
                                        <?php if ((int)$code >= 400 && (int)$code < 500): ?>
                                        <div class="http-code-item">
                                            <span class="http-code-badge client-error"><?= $code ?></span>
                                            <span class="http-code-count"><?= number_format($count) ?></span>
                                        </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="http-category server-error">
                                <span class="http-category-label">5xx Server</span>
                                <div class="http-codes-list" id="http5xx">
                                    <?php foreach ($httpCodes as $code => $count): ?>
                                        <?php if ((int)$code >= 500): ?>
                                        <div class="http-code-item">
                                            <span class="http-code-badge server-error"><?= $code ?></span>
                                            <span class="http-code-count"><?= number_format($count) ?></span>
                                        </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- v3.1: Enhanced Top Domains -->
                <div class="section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <circle cx="12" cy="12" r="10"/>
                                <line x1="2" y1="12" x2="22" y2="12"/>
                                <path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/>
                            </svg>
                            Domains (<span id="domainCount"><?= $totalDomains ?></span>)
                        </h2>
                    </div>
                    <div class="section-body" id="topDomains">
                        <div class="domain-health-summary">
                            <span class="health-indicator healthy" id="healthyDomains">🟢 0</span>
                            <span class="health-indicator problematic" id="problematicDomains">🔴 0</span>
                            <span class="health-indicator slow" id="slowDomains">🟡 0</span>
                        </div>
                        <div class="domain-list-enhanced" id="domainList">
                        <?php
                        $sortedDomains = $domainStats;
                        uasort($sortedDomains, fn($a, $b) => ($b['total'] ?? 0) <=> ($a['total'] ?? 0));
                        $healthyCount = 0;
                        $problematicCount = 0;
                        $slowCount = 0;
                        foreach (array_slice($sortedDomains, 0, 10, true) as $domain => $stats):
                            $domainTotal = $stats['total'] ?? 0;
                            $domainSuccess = $stats['success'] ?? 0;
                            $domainError = $stats['error'] ?? 0;
                            $domainPercent = $domainTotal > 0 ? round(($domainSuccess / $domainTotal) * 100) : 0;
                            $errorRate = $domainTotal > 0 ? ($domainError / $domainTotal) : 0;
                            $healthClass = $errorRate > 0.5 ? 'problematic' : ($errorRate > 0.2 ? 'slow' : 'healthy');
                            if ($errorRate > 0.5) $problematicCount++;
                            elseif ($errorRate > 0.2) $slowCount++;
                            else $healthyCount++;
                        ?>
                        <div class="domain-item-enhanced <?= $healthClass ?>">
                            <div class="domain-icon"><?= strtoupper(substr($domain, 0, 2)) ?></div>
                            <div class="domain-info">
                                <div class="domain-name"><?= htmlspecialchars($domain) ?></div>
                                <div class="domain-meta">
                                    <div class="domain-bar">
                                        <div class="domain-bar-fill" style="width: <?= $domainPercent ?>%"></div>
                                    </div>
                                    <span class="domain-stats"><?= $domainSuccess ?>/<?= $domainTotal ?> (<?= $domainPercent ?>%)</span>
                                </div>
                            </div>
                            <?php if ($domainError > 0): ?>
                            <span class="domain-error-badge"><?= $domainError ?> err</span>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                        <?php if (empty($domainStats)): ?>
                        <div class="empty-state">
                            <p class="empty-state-text">Waiting for data...</p>
                        </div>
                        <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Toast Container -->
    <div id="toastContainer" class="toast-container"></div>

    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
    <script>
    (function() {
        'use strict';

        const API_BASE = '<?= $baseUrl ?>/api/v1';
        const CSRF_TOKEN = '<?= htmlspecialchars($csrfToken) ?>';
        const PROCESS_ID = '<?= htmlspecialchars($processId) ?>';
        const POLL_INTERVAL = 2000;
        const IS_INITIALIZING = <?= $isInitializing ? 'true' : 'false' ?>;

        let isComplete = <?= $isComplete ? 'true' : 'false' ?>;
        let isPaused = <?= $isPaused ? 'true' : 'false' ?>;
        let isInitialized = !IS_INITIALIZING;
        let pollTimer = null;
        let processingStartTime = Date.now();

        // Speed Chart
        const speedHistory = [];
        const speedLabels = [];
        const MAX_SPEED_POINTS = 30;
        let speedChart = null;
        let lastProcessed = <?= $processedLinks ?>;
        let lastTime = Date.now();

        // v3.1: Initialize process if needed
        async function initializeProcess() {
            if (!IS_INITIALIZING) return true;

            console.log('[Init] Starting process initialization...');
            updateInitStep('stepExtract', 'loading');

            try {
                const response = await fetch(`${API_BASE}/process/${PROCESS_ID}/initialize`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-Token': CSRF_TOKEN
                    }
                });

                const result = await response.json();
                console.log('[Init] Result:', result);

                if (result.success) {
                    updateInitStep('stepExtract', 'done');
                    updateInitStep('stepInit', 'done');
                    updateInitStep('stepCreate', 'done');

                    // Wait a moment for visual feedback
                    await new Promise(r => setTimeout(r, 500));

                    // Hide overlay, show main content
                    hideInitializingOverlay();
                    isInitialized = true;

                    // CRITICAL FIX: Reset processingStartTime AFTER initialization
                    // so the 5-second minimum starts from when user sees processing view
                    processingStartTime = Date.now();

                    // v3.1 FIX: Update status to Running
                    const statusIndicator = document.getElementById('statusIndicator');
                    const statusText = document.getElementById('statusText');
                    if (statusIndicator) statusIndicator.className = 'status-indicator running';
                    if (statusText) statusText.textContent = 'Running';

                    showToast('Process started with ' + result.total_links + ' URLs', 'success');
                    console.log('[Init] ✅ Initialization complete, starting processing...');
                    return true;
                } else {
                    console.error('[Init] ❌ Initialization failed:', result.error);
                    showToast('Error: ' + (result.error || 'Initialization failed'), 'error');
                    return false;
                }
            } catch (error) {
                console.error('[Init] Error:', error);
                showToast('Connection error', 'error');
                return false;
            }
        }

        function updateInitStep(stepId, status) {
            const step = document.getElementById(stepId);
            if (!step) return;

            step.classList.remove('active');

            if (status === 'loading') {
                step.classList.add('active');
                step.querySelector('.step-icon').innerHTML = `
                    <svg class="spinner-svg" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="18" height="18">
                        <circle cx="12" cy="12" r="10" stroke-opacity="0.25"/>
                        <path d="M12 2a10 10 0 0 1 10 10" stroke-linecap="round"/>
                    </svg>`;
            } else if (status === 'done') {
                step.classList.add('done');
                step.querySelector('.step-icon').innerHTML = `
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" width="18" height="18">
                        <polyline points="20 6 9 17 4 12"/>
                    </svg>`;
            }
        }

        function hideInitializingOverlay() {
            const overlay = document.getElementById('initializingOverlay');
            const mainContent = document.getElementById('mainContent');

            if (overlay) {
                overlay.style.opacity = '0';
                setTimeout(() => overlay.remove(), 300);
            }

            if (mainContent) {
                mainContent.style.opacity = '1';
                mainContent.style.pointerEvents = 'auto';
            }
        }

        function initSpeedChart() {
            const ctx = document.getElementById('speedChart');
            if (!ctx || !window.Chart) return;

            speedChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: speedLabels,
                    datasets: [{
                        label: 'URLs/s',
                        data: speedHistory,
                        borderColor: 'rgba(59, 130, 246, 1)',
                        backgroundColor: 'rgba(59, 130, 246, 0.1)',
                        fill: true,
                        tension: 0.4,
                        pointRadius: 0,
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: { duration: 0 },
                    plugins: { legend: { display: false } },
                    scales: {
                        x: { display: false },
                        y: {
                            beginAtZero: true,
                            ticks: { color: 'rgba(255,255,255,0.5)', font: { size: 10 } },
                            grid: { color: 'rgba(255,255,255,0.05)' }
                        }
                    }
                }
            });
        }

        function updateSpeedChart(currentProcessed) {
            const now = Date.now();
            const timeDiff = (now - lastTime) / 1000;
            const processedDiff = currentProcessed - lastProcessed;

            if (timeDiff > 0.5) {
                const speed = Math.round((processedDiff / timeDiff) * 10) / 10;

                speedHistory.push(speed);
                speedLabels.push('');

                if (speedHistory.length > MAX_SPEED_POINTS) {
                    speedHistory.shift();
                    speedLabels.shift();
                }

                if (speedChart) speedChart.update('none');

                const avgSpeed = speedHistory.reduce((a, b) => a + b, 0) / speedHistory.length;
                const peakSpeed = Math.max(...speedHistory);

                const avgEl = document.getElementById('avgSpeed');
                const peakEl = document.getElementById('peakSpeed');
                if (avgEl) avgEl.textContent = avgSpeed.toFixed(1) + '/s';
                if (peakEl) peakEl.textContent = peakSpeed.toFixed(1) + '/s';

                lastProcessed = currentProcessed;
                lastTime = now;
            }
        }

        async function apiCall(endpoint, method = 'GET', data = null) {
            const options = {
                method,
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-Token': CSRF_TOKEN
                }
            };
            if (data && method !== 'GET') {
                options.body = JSON.stringify(data);
            }
            try {
                const response = await fetch(`${API_BASE}/${endpoint}`, options);
                return await response.json();
            } catch (error) {
                console.error('API Error:', error);
                return { success: false, error: { message: error.message } };
            }
        }

        function showToast(message, type = 'info') {
            const container = document.getElementById('toastContainer');
            if (!container) return;
            const toast = document.createElement('div');
            toast.className = `toast toast-${type}`;
            toast.innerHTML = `<span>${message}</span>`;
            container.appendChild(toast);
            setTimeout(() => toast.remove(), 4000);
        }

        let prevStats = {
            imported: 0, ignored: 0, errors: 0, remaining: 0, successRate: 0,
            retries: { total: 0, p0: 0, p1: 0, p2: 0, p3: 0 }
        };

        function updateProgress(data) {
            if (!data) return;

            const total = Math.max(data.total_links || 1, 1);
            const imported = data.imported_links || 0;
            const ignored = data.ignored_links || 0;
            const errors = data.error_links || 0;
            const processed = imported + ignored + errors;
            const remaining = total - processed;
            const percent = Math.round((processed / total) * 100 * 10) / 10;

            // Update circle
            const circle = document.getElementById('progressCircle');
            if (circle) circle.style.strokeDashoffset = 276.46 - (276.46 * percent / 100);

            const progressValue = document.getElementById('progressValue');
            if (progressValue) progressValue.textContent = percent + '%';

            // Animate hero stats
            animateStatUpdate('statImported', prevStats.imported, imported);
            animateStatUpdate('statIgnored', prevStats.ignored, ignored);
            animateStatUpdate('statErrors', prevStats.errors, errors);
            animateStatUpdate('statRemaining', prevStats.remaining, remaining);

            prevStats.imported = imported;
            prevStats.ignored = ignored;
            prevStats.errors = errors;
            prevStats.remaining = remaining;

            // Update speed chart
            updateSpeedChart(processed);
            const currentSpeed = speedHistory.length > 0 ? speedHistory[speedHistory.length - 1] : 0;

            // Performance live
            updatePerformanceLive(data, currentSpeed);

            // v3.1: Update detailed stats
            updateDetailedStats(data);

            // v3.1: Update request stats
            updateRequestStats(data);

            // Circuit breaker
            updateCircuitBreaker(data);

            // Retry stats
            updateRetryStats(data);

            // Phase details
            updatePhaseDetails(data);

            // v3.1: Update HTTP codes by category
            updateHttpCodesCategorized(data);

            // v3.1: Update domains with health
            updateDomainsWithHealth(data);

            // Times
            const timeElapsed = document.getElementById('timeElapsed');
            if (timeElapsed) timeElapsed.textContent = formatTime(data.elapsed_time || 0);

            const timeRemaining = document.getElementById('timeRemaining');
            if (timeRemaining) timeRemaining.textContent = formatTime(data.remaining_time || 0);

            const processingRate = document.getElementById('processingRate');
            if (processingRate) processingRate.textContent = currentSpeed.toFixed(1) + '/s';

            // Check complete - show in-place, don't reload immediately
            if (remaining === 0 && total > 0 && !isComplete) {
                // Minimum 5 seconds of processing view
                const elapsedMs = Date.now() - processingStartTime;
                if (elapsedMs < 5000) {
                    return; // Wait for minimum display time
                }

                isComplete = true;
                stopPolling();
                showNotification('Processing Complete', `Processed ${total} URLs`);
                showCompletionBanner(total, imported, ignored, errors); // Show banner in-place
            }
        }

        function animateStatUpdate(elementId, prevValue, newValue) {
            const element = document.getElementById(elementId);
            if (!element || prevValue === newValue) return;
            animateValue(element, prevValue, newValue);
        }

        function formatTime(seconds) {
            const h = Math.floor(seconds / 3600);
            const m = Math.floor((seconds % 3600) / 60);
            const s = Math.floor(seconds % 60);
            return `${h.toString().padStart(2, '0')}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
        }

        function easeOutCubic(x) { return 1 - Math.pow(1 - x, 3); }

        function animateValue(element, start, end, duration = 250) {
            if (!element || start === end) return;
            const range = end - start;
            const startTime = performance.now();

            function update(currentTime) {
                const elapsed = currentTime - startTime;
                const progress = Math.min(elapsed / duration, 1);
                const current = Math.round(start + range * easeOutCubic(progress));
                element.textContent = current.toLocaleString();
                if (progress < 1) requestAnimationFrame(update);
            }
            requestAnimationFrame(update);
        }

        function updatePerformanceLive(data, currentSpeed) {
            const perfUrlsPerSecond = document.getElementById('perfUrlsPerSecond');
            if (perfUrlsPerSecond) perfUrlsPerSecond.textContent = currentSpeed.toFixed(1);

            const perfSpeedBar = document.getElementById('perfSpeedBar');
            if (perfSpeedBar) perfSpeedBar.style.width = Math.min((currentSpeed / 100) * 100, 100) + '%';

            const batchesProcessed = data.batches_processed || 0;
            const currentPhase = data.phase_index || 0;
            const phaseQueues = data.phase_queues || {};
            const currentQueueSize = (phaseQueues[currentPhase] || []).length;

            const perfBatchCurrent = document.getElementById('perfBatchCurrent');
            const perfBatchTotal = document.getElementById('perfBatchTotal');
            const perfBatchBar = document.getElementById('perfBatchBar');

            if (perfBatchCurrent) perfBatchCurrent.textContent = batchesProcessed;
            if (perfBatchTotal) perfBatchTotal.textContent = currentQueueSize > 0 ? currentQueueSize : '-';
            if (perfBatchBar && currentQueueSize > 0) {
                perfBatchBar.style.width = Math.min(((data.total_links - currentQueueSize) / data.total_links) * 100, 100) + '%';
            }

            const requestStats = data.request_stats || {};
            const totalRequests = requestStats.total_requests || 0;
            const totalResponseTime = requestStats.total_response_time || 0;
            const avgLatency = totalRequests > 0 ? Math.round((totalResponseTime / totalRequests) * 1000) : 0;

            const perfLatency = document.getElementById('perfLatency');
            const perfLatencyBar = document.getElementById('perfLatencyBar');

            const displayLatency = avgLatency > 0 ? avgLatency : (currentSpeed > 0 ? Math.round(1000 / currentSpeed) : 0);
            if (perfLatency) perfLatency.textContent = displayLatency;
            if (perfLatencyBar) perfLatencyBar.style.width = Math.min((displayLatency / 500) * 100, 100) + '%';

            const total = data.total_links || 1;
            const imported = data.imported_links || 0;
            const newSuccessRate = total > 0 ? ((imported / total) * 100).toFixed(1) : 0;

            const perfSuccessRate = document.getElementById('perfSuccessRate');
            const perfSuccessBar = document.getElementById('perfSuccessBar');

            if (perfSuccessRate) perfSuccessRate.textContent = newSuccessRate;
            if (perfSuccessBar) perfSuccessBar.style.width = newSuccessRate + '%';
        }

        // v3.1: Update detailed stats
        function updateDetailedStats(data) {
            const ignoredDetails = data.ignored_details || {};
            const errorDetails = data.error_details || {};

            // Calculate totals for percentage bars
            const ignoredTotal = (ignoredDetails.duplicate || 0) + (ignoredDetails.invalid_url || 0) +
                                (ignoredDetails.blocked_domain || 0) + (ignoredDetails.non_html || 0);
            const errorTotal = (errorDetails.http_403 || 0) + (errorDetails.http_404 || 0) +
                              (errorDetails.http_429 || 0) + (errorDetails.http_5xx || 0) +
                              (errorDetails.timeout || 0) + (errorDetails.connection || 0);

            // Update ignored breakdown
            updateBreakdownItem('ignoredDuplicate', 'ignoredDuplicateVal', ignoredDetails.duplicate || 0, ignoredTotal);
            updateBreakdownItem('ignoredInvalid', 'ignoredInvalidVal', ignoredDetails.invalid_url || 0, ignoredTotal);
            updateBreakdownItem('ignoredBlocked', 'ignoredBlockedVal', ignoredDetails.blocked_domain || 0, ignoredTotal);
            updateBreakdownItem('ignoredNonHtml', 'ignoredNonHtmlVal', ignoredDetails.non_html || 0, ignoredTotal);

            // Update error breakdown
            updateBreakdownItem('error403', 'error403Val', errorDetails.http_403 || 0, errorTotal);
            updateBreakdownItem('error404', 'error404Val', errorDetails.http_404 || 0, errorTotal);
            updateBreakdownItem('error429', 'error429Val', errorDetails.http_429 || 0, errorTotal);
            updateBreakdownItem('error5xx', 'error5xxVal', errorDetails.http_5xx || 0, errorTotal);
            updateBreakdownItem('errorTimeout', 'errorTimeoutVal', errorDetails.timeout || 0, errorTotal);
            updateBreakdownItem('errorConnection', 'errorConnectionVal', errorDetails.connection || 0, errorTotal);
        }

        function updateBreakdownItem(barId, valId, value, total) {
            const bar = document.getElementById(barId);
            const val = document.getElementById(valId);
            if (bar) bar.style.width = (total > 0 ? (value / total) * 100 : 0) + '%';
            if (val) val.textContent = value;
        }

        // v3.1: Update request stats with percentiles
        function updateRequestStats(data) {
            const requestStats = data.request_stats || {};

            const totalRequests = document.getElementById('totalRequests');
            const totalResponseTime = document.getElementById('totalResponseTime');

            if (totalRequests) totalRequests.textContent = (requestStats.total_requests || 0).toLocaleString();
            if (totalResponseTime) totalResponseTime.textContent = (requestStats.total_response_time || 0).toFixed(1) + 's';

            // Calculate percentiles
            const times = requestStats.response_times || [];
            if (times.length > 0) {
                const sorted = [...times].sort((a, b) => a - b);
                const p50El = document.getElementById('p50');
                const p90El = document.getElementById('p90');
                const p99El = document.getElementById('p99');

                if (p50El) p50El.textContent = Math.round(sorted[Math.floor(sorted.length * 0.5)] * 1000) + 'ms';
                if (p90El) p90El.textContent = Math.round(sorted[Math.floor(sorted.length * 0.9)] * 1000) + 'ms';
                if (p99El) p99El.textContent = Math.round(sorted[Math.floor(sorted.length * 0.99)] * 1000) + 'ms';
            }
        }

        function updateCircuitBreaker(data) {
            const domainStats = data.domain_stats || {};
            const problematicDomains = [];

            for (const [domain, stats] of Object.entries(domainStats)) {
                const total = stats.total || 0;
                const errors = stats.error || 0;
                if (total >= 3 && (errors / total) >= 0.5) {
                    problematicDomains.push(domain);
                }
            }

            const cbIndicator = document.getElementById('cbIndicator');
            const cbStatusText = document.getElementById('cbStatusText');
            const cbBlockedDomains = document.getElementById('cbBlockedDomains');
            const cbBlockedList = document.getElementById('cbBlockedList');
            const cbBlockedDomainsList = document.getElementById('cbBlockedDomainsList');

            const blockedCount = problematicDomains.length;
            if (cbBlockedDomains) cbBlockedDomains.textContent = blockedCount;

            if (blockedCount === 0) {
                if (cbIndicator) cbIndicator.className = 'cb-indicator cb-closed';
                if (cbStatusText) cbStatusText.textContent = 'Closed (OK)';
                if (cbBlockedList) cbBlockedList.style.display = 'none';
            } else {
                if (cbIndicator) cbIndicator.className = 'cb-indicator cb-open';
                if (cbStatusText) cbStatusText.textContent = 'Open (' + blockedCount + ' problematic)';
                if (cbBlockedList) cbBlockedList.style.display = 'block';
                if (cbBlockedDomainsList) {
                    cbBlockedDomainsList.innerHTML = problematicDomains.slice(0, 5)
                        .map(d => '<li>' + d + '</li>').join('');
                }
            }
        }

        function updateRetryStats(data) {
            const retryStats = data.retry_stats || { total_retries: 0, retries_by_phase: {0:0,1:0,2:0,3:0} };
            const totalRetries = retryStats.total_retries || 0;
            const retriesByPhase = retryStats.retries_by_phase || {0:0,1:0,2:0,3:0};

            const retryTotal = document.getElementById('retryTotal');
            if (retryTotal) retryTotal.textContent = totalRetries;

            const retrySuccessRate = document.getElementById('retrySuccessRate');
            if (retrySuccessRate) {
                const imported = data.imported_links || 0;
                const urlsWithRetries = retryStats.urls_with_retries || 0;
                const successPercent = urlsWithRetries > 0
                    ? Math.round((imported / Math.max(imported + urlsWithRetries, 1)) * 100)
                    : (imported > 0 ? 100 : 0);
                retrySuccessRate.textContent = successPercent + '%';
            }

            const phaseValues = [retriesByPhase[0]||0, retriesByPhase[1]||0, retriesByPhase[2]||0, retriesByPhase[3]||0];
            const maxRetries = Math.max(...phaseValues, 1);

            for (let i = 0; i <= 3; i++) {
                const phaseCount = retriesByPhase[i] || 0;
                const phaseBar = document.getElementById('retryPhase' + i + 'Bar');
                const phaseCountEl = document.getElementById('retryPhase' + i);

                if (phaseBar) phaseBar.style.width = ((phaseCount / maxRetries) * 100) + '%';
                if (phaseCountEl) phaseCountEl.textContent = phaseCount;
            }
        }

        function updatePhaseDetails(data) {
            const phaseStats = data.phase_stats || {};
            const phaseQueues = data.phase_queues || {};
            const currentPhase = data.phase_index || 0;

            for (let i = 0; i <= 3; i++) {
                const stats = phaseStats[i] || { success: 0, retry: 0, error: 0 };
                const queueCount = (phaseQueues[i] || []).length;

                const processedEl = document.getElementById('phaseProcessed' + i);
                const queueEl = document.getElementById('phaseQueue' + i);
                const speedEl = document.getElementById('phaseSpeed' + i);
                const detailsEl = document.getElementById('phaseDetails' + i);

                if (processedEl) processedEl.textContent = stats.success;
                if (queueEl) queueEl.textContent = queueCount;

                if (speedEl) {
                    if (i === currentPhase && speedHistory.length > 0) {
                        speedEl.textContent = speedHistory[speedHistory.length - 1].toFixed(1) + '/s';
                    } else {
                        speedEl.textContent = '-/s';
                    }
                }

                if (detailsEl) {
                    if (i === currentPhase) detailsEl.classList.add('expanded');
                    else detailsEl.classList.remove('expanded');
                }
            }
        }

        // v3.1: Update HTTP codes by category
        function updateHttpCodesCategorized(data) {
            const httpCodes = data.http_codes || {};
            const categories = { '2xx': [], '3xx': [], '4xx': [], '5xx': [] };

            for (const [code, count] of Object.entries(httpCodes)) {
                const codeNum = parseInt(code);
                if (codeNum >= 200 && codeNum < 300) categories['2xx'].push({ code, count });
                else if (codeNum >= 300 && codeNum < 400) categories['3xx'].push({ code, count });
                else if (codeNum >= 400 && codeNum < 500) categories['4xx'].push({ code, count });
                else if (codeNum >= 500) categories['5xx'].push({ code, count });
            }

            for (const [cat, codes] of Object.entries(categories)) {
                const container = document.getElementById('http' + cat);
                if (container) {
                    container.innerHTML = codes.map(c => `
                        <div class="http-code-item">
                            <span class="http-code-badge ${cat === '2xx' ? 'success' : (cat === '3xx' ? 'redirect' : (cat === '4xx' ? 'client-error' : 'server-error'))}">${c.code}</span>
                            <span class="http-code-count">${c.count.toLocaleString()}</span>
                        </div>
                    `).join('') || '<span class="empty-category">-</span>';
                }
            }
        }

        // v3.1: Update domains with health indicators
        function updateDomainsWithHealth(data) {
            const domainStats = data.domain_stats || {};
            const sortedDomains = Object.entries(domainStats).sort((a, b) => (b[1].total || 0) - (a[1].total || 0));

            let healthyCount = 0, problematicCount = 0, slowCount = 0;

            const domainList = document.getElementById('domainList');
            if (domainList) {
                domainList.innerHTML = sortedDomains.slice(0, 10).map(([domain, stats]) => {
                    const total = stats.total || 0;
                    const success = stats.success || 0;
                    const error = stats.error || 0;
                    const percent = total > 0 ? Math.round((success / total) * 100) : 0;
                    const errorRate = total > 0 ? (error / total) : 0;

                    let healthClass = 'healthy';
                    if (errorRate > 0.5) { healthClass = 'problematic'; problematicCount++; }
                    else if (errorRate > 0.2) { healthClass = 'slow'; slowCount++; }
                    else { healthyCount++; }

                    return `
                        <div class="domain-item-enhanced ${healthClass}">
                            <div class="domain-icon">${domain.substring(0, 2).toUpperCase()}</div>
                            <div class="domain-info">
                                <div class="domain-name">${domain}</div>
                                <div class="domain-meta">
                                    <div class="domain-bar"><div class="domain-bar-fill" style="width: ${percent}%"></div></div>
                                    <span class="domain-stats">${success}/${total} (${percent}%)</span>
                                </div>
                            </div>
                            ${error > 0 ? `<span class="domain-error-badge">${error} err</span>` : ''}
                        </div>
                    `;
                }).join('') || '<div class="empty-state"><p class="empty-state-text">Waiting for data...</p></div>';
            }

            const domainCount = document.getElementById('domainCount');
            if (domainCount) domainCount.textContent = sortedDomains.length;

            const healthyEl = document.getElementById('healthyDomains');
            const problematicEl = document.getElementById('problematicDomains');
            const slowEl = document.getElementById('slowDomains');

            if (healthyEl) healthyEl.textContent = '🟢 ' + healthyCount;
            if (problematicEl) problematicEl.textContent = '🔴 ' + problematicCount;
            if (slowEl) slowEl.textContent = '🟡 ' + slowCount;
        }

        function showNotification(title, body) {
            if ('Notification' in window && Notification.permission === 'granted') {
                new Notification(title, { body });
            }
        }

        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission();
        }

        // Show completion banner in-place (no page reload)
        function showCompletionBanner(total, imported, ignored, errors) {
            const successRate = total > 0 ? ((imported / total) * 100).toFixed(1) : 0;
            const timeElapsed = document.getElementById('timeElapsed')?.textContent || '00:00:00';

            // Update status indicator
            const statusIndicator = document.getElementById('statusIndicator');
            const statusText = document.getElementById('statusText');
            if (statusIndicator) statusIndicator.className = 'status-indicator completed';
            if (statusText) statusText.textContent = 'Completed';

            // Update progress circle to 100%
            const progressCircle = document.getElementById('progressCircle');
            const progressValue = document.getElementById('progressValue');
            if (progressCircle) {
                progressCircle.style.strokeDashoffset = '0';
                progressCircle.classList.add('complete');
            }
            if (progressValue) progressValue.textContent = '100%';

            // Create and show completion banner
            const progressHero = document.getElementById('progressHero');
            if (progressHero) {
                const banner = document.createElement('div');
                banner.className = 'completion-banner';
                banner.innerHTML = `
                    <div class="completion-icon">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="32" height="32">
                            <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                            <polyline points="22 4 12 14.01 9 11.01"/>
                        </svg>
                    </div>
                    <div class="completion-info">
                        <h2 class="completion-title">Processing Complete!</h2>
                        <p class="completion-subtitle">
                            Processed ${total.toLocaleString()} URLs in ${timeElapsed}
                            with ${successRate}% success rate
                        </p>
                    </div>
                    <div class="completion-actions">
                        <a href="<?= $baseUrl ?>/?export=<?= urlencode($processId) ?>" class="btn btn-primary">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                <polyline points="7 10 12 15 17 10"/>
                                <line x1="12" y1="15" x2="12" y2="3"/>
                            </svg>
                            Export Results
                        </a>
                        <a href="<?= $baseUrl ?>/new" class="btn btn-secondary">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                                <circle cx="12" cy="12" r="10"/>
                                <line x1="12" y1="8" x2="12" y2="16"/>
                                <line x1="8" y1="12" x2="16" y2="12"/>
                            </svg>
                            New Process
                        </a>
                    </div>
                `;
                progressHero.insertAdjacentElement('beforebegin', banner);
            }

            // Hide controls section (Pause/Cancel buttons)
            const controlButtons = document.getElementById('controlButtons');
            if (controlButtons) controlButtons.style.display = 'none';
        }

        async function pollProgress() {
            if (isComplete || !isInitialized) return;
            const result = await apiCall(`process/${PROCESS_ID}/live`);
            if (result.success) updateProgress(result.data);
        }

        function startPolling() {
            if (pollTimer) clearInterval(pollTimer);
            pollTimer = setInterval(pollProgress, POLL_INTERVAL);
            pollProgress();
        }

        function stopPolling() {
            if (pollTimer) { clearInterval(pollTimer); pollTimer = null; }
        }

        document.addEventListener('visibilitychange', () => {
            if (document.hidden) stopPolling();
            else if (!isComplete && isInitialized) startPolling();
        });

        let isProcessing = false;
        let processingLoopCount = 0;

        async function runProcessingLoop() {
            if (isProcessing || isComplete || isPaused || !isInitialized) {
                console.log('[Processing] ⛔ Loop blocked - isProcessing:', isProcessing, 'isComplete:', isComplete, 'isPaused:', isPaused, 'isInitialized:', isInitialized);
                return;
            }
            isProcessing = true;
            console.log('[Processing] 🚀 Starting AJAX processing loop...');

            while (!isComplete && !isPaused && isInitialized) {
                processingLoopCount++;
                try {
                    console.log(`[Processing] 📤 Loop #${processingLoopCount} - Calling /run API...`);

                    const response = await fetch(`${API_BASE}/process/${PROCESS_ID}/run`, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json', 'X-CSRF-Token': CSRF_TOKEN }
                    });

                    if (!response.ok) {
                        console.error(`[Processing] ❌ HTTP Error: ${response.status} ${response.statusText}`);
                        await new Promise(r => setTimeout(r, 2000));
                        continue;
                    }

                    const result = await response.json();
                    console.log('[Processing] 📥 Response:', result.success ? 'success' : 'failed', 'status:', result.status, 'processed:', result.data?.processed_links || 0);

                    if (!result.success) {
                        console.error('[Processing] ❌ API Error:', result.error);
                        if (result.error === 'Process not found') {
                            showToast('Process not found', 'error');
                            break;
                        }
                        await new Promise(r => setTimeout(r, 2000));
                        continue;
                    }

                    if (result.data) updateProgress(result.data);

                    // v3.1 FIX: Append new logs to live log in real-time
                    if (result.logs && result.logs.length > 0) {
                        appendLiveLogs(result.logs);
                    }

                    if (result.complete || result.status === 'completed') {
                        // CRITICAL FIX: Ensure minimum 5-second display time
                        // so users can actually SEE the live processing
                        const elapsedMs = Date.now() - processingStartTime;
                        if (elapsedMs < 5000) {
                            // Keep showing progress updates for at least 5 seconds
                            console.log('[Processing] ⏳ Complete but waiting for minimum display time...');
                            await new Promise(r => setTimeout(r, 500));
                            continue;  // Don't break yet, keep polling
                        }

                        console.log('[Processing] ✅ Complete!');
                        isComplete = true;
                        stopPolling();
                        showNotification('Processing Complete', `Processed ${result.data?.total_links || 0} URLs`);
                        showCompletionBanner(
                            result.data?.total_links || 0,
                            result.data?.imported_links || 0,
                            result.data?.ignored_links || 0,
                            result.data?.error_links || 0
                        );
                        break;
                    }

                    if (result.status === 'paused') { console.log('[Processing] ⏸ Paused'); isPaused = true; break; }
                    if (result.status === 'cancelled') { console.log('[Processing] 🛑 Cancelled'); isComplete = true; break; }

                    await new Promise(r => setTimeout(r, 100));
                } catch (error) {
                    console.error('[Processing] 💥 Exception:', error);
                    await new Promise(r => setTimeout(r, 2000));
                }
            }

            isProcessing = false;
            console.log(`[Processing] 🏁 Loop ended after ${processingLoopCount} iterations`);
        }

        // v3.1: Real-time log appending
        let totalLogCount = <?= count($logs) ?>;

        function appendLiveLogs(logs) {
            const liveLog = document.getElementById('liveLog');
            if (!liveLog || !logs || logs.length === 0) return;

            // Remove empty state if present
            const emptyState = liveLog.querySelector('.empty-state');
            if (emptyState) emptyState.remove();

            // Create and prepend log entries (newest first)
            logs.forEach(log => {
                const entry = document.createElement('div');
                entry.className = `log-entry-enhanced log-${log.class || 'info'}`;
                entry.dataset.type = log.class || 'info';

                const timestamp = log.timestamp ? new Date(log.timestamp).toLocaleTimeString('en-US', {hour12: false}) : '--:--:--';
                const httpCode = log.http_code || '-';
                const responseTime = log.response_time ? Math.round(log.response_time * 1000) : 0;
                const domain = log.domain || (log.url ? new URL(log.url).hostname : '-');
                const statusIcon = log.class === 'success' ? '✓' : (log.class === 'error' ? '✗' : (log.class === 'warning' ? '○' : '•'));
                const codeClass = httpCode >= 200 && httpCode < 300 ? 'success' : (httpCode >= 300 && httpCode < 400 ? 'redirect' : (httpCode >= 400 && httpCode < 500 ? 'client-error' : 'server-error'));

                entry.innerHTML = `
                    <div class="log-row-main">
                        <span class="log-timestamp">${timestamp}</span>
                        <span class="log-status-icon log-${log.class || 'info'}">${statusIcon}</span>
                        <span class="log-http-code ${codeClass}">${httpCode}</span>
                        <span class="log-response-time">${responseTime}ms</span>
                        <span class="log-domain">${escapeHtml(domain)}</span>
                    </div>
                    <div class="log-row-url">${escapeHtml(log.url || '')}</div>
                    ${log.message ? `<div class="log-row-message">${escapeHtml(log.message)}</div>` : ''}
                `;

                // Add animation class
                entry.style.opacity = '0';
                entry.style.transform = 'translateY(-10px)';

                // Prepend to show newest at top
                liveLog.insertBefore(entry, liveLog.firstChild);

                // Trigger animation
                requestAnimationFrame(() => {
                    entry.style.transition = 'opacity 0.3s, transform 0.3s';
                    entry.style.opacity = '1';
                    entry.style.transform = 'translateY(0)';
                });

                totalLogCount++;
            });

            // Update log count
            const logCountEl = document.getElementById('logCount');
            if (logCountEl) logCountEl.textContent = totalLogCount + ' entries';

            // Limit entries to prevent memory issues (keep last 200)
            const entries = liveLog.querySelectorAll('.log-entry-enhanced');
            if (entries.length > 200) {
                for (let i = 200; i < entries.length; i++) {
                    entries[i].remove();
                }
            }

            // Apply current filter
            filterLogs();

            console.log(`[Logs] 📝 Added ${logs.length} new log entries (total: ${totalLogCount})`);
        }

        function escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }

        // Log filter functionality
        let currentLogFilter = 'all';
        const logSearch = document.getElementById('logSearch');
        const filterBtns = document.querySelectorAll('.filter-btn');

        filterBtns.forEach(btn => {
            btn.addEventListener('click', () => {
                filterBtns.forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                currentLogFilter = btn.dataset.filter;
                filterLogs();
            });
        });

        if (logSearch) {
            logSearch.addEventListener('input', debounce(filterLogs, 300));
        }

        function filterLogs() {
            const searchTerm = logSearch ? logSearch.value.toLowerCase() : '';
            const entries = document.querySelectorAll('.log-entry-enhanced');

            entries.forEach(entry => {
                const type = entry.dataset.type;
                const text = entry.textContent.toLowerCase();

                const matchesFilter = currentLogFilter === 'all' || type === currentLogFilter;
                const matchesSearch = !searchTerm || text.includes(searchTerm);

                entry.style.display = matchesFilter && matchesSearch ? 'block' : 'none';
            });
        }

        function debounce(fn, delay) {
            let timeout;
            return function(...args) {
                clearTimeout(timeout);
                timeout = setTimeout(() => fn.apply(this, args), delay);
            };
        }

        // Controls
        const btnPause = document.getElementById('btnPause');
        const btnResume = document.getElementById('btnResume');
        const btnCancel = document.getElementById('btnCancel');

        if (btnPause) {
            btnPause.addEventListener('click', async () => {
                btnPause.disabled = true;
                const result = await apiCall(`process/${PROCESS_ID}/pause`, 'POST');
                if (result.success) { showToast('Process paused', 'warning'); location.reload(); }
                else { showToast(result.error?.message || 'Failed', 'error'); btnPause.disabled = false; }
            });
        }

        if (btnResume) {
            btnResume.addEventListener('click', async () => {
                btnResume.disabled = true;
                const result = await apiCall(`process/${PROCESS_ID}/resume`, 'POST');
                if (result.success) { showToast('Process resumed', 'success'); location.reload(); }
                else { showToast(result.error?.message || 'Failed', 'error'); btnResume.disabled = false; }
            });
        }

        if (btnCancel) {
            btnCancel.addEventListener('click', async () => {
                if (!confirm('Cancel process?')) return;
                btnCancel.disabled = true;
                const result = await apiCall(`process/${PROCESS_ID}/cancel`, 'POST');
                if (result.success) {
                    showToast('Process cancelled', 'error');
                    stopPolling();
                    setTimeout(() => { window.location.href = '<?= $baseUrl ?>/'; }, 1500);
                } else {
                    showToast(result.error?.message || 'Failed', 'error');
                    btnCancel.disabled = false;
                }
            });
        }

        // Init
        async function main() {
            console.log('[Main] 🎬 Starting main() - IS_INITIALIZING:', IS_INITIALIZING, 'isComplete:', isComplete);
            initSpeedChart();

            if (IS_INITIALIZING) {
                console.log('[Main] 📦 Calling initializeProcess()...');
                const success = await initializeProcess();
                console.log('[Main] 📦 initializeProcess() returned:', success);
                if (success) {
                    console.log('[Main] ▶️ Starting polling and processing loop...');
                    startPolling();
                    runProcessingLoop();
                } else {
                    console.error('[Main] ❌ Initialization failed, not starting processing');
                }
            } else if (!isComplete) {
                console.log('[Main] ⏩ Already initialized, starting polling and processing loop...');
                startPolling();
                runProcessingLoop();
            } else {
                console.log('[Main] ✅ Process already complete');
            }
        }

        main();
    })();
    </script>
</body>
</html>
