// Sistema Masonry para layout Pinterest - Baseado no DigUpDOG
//masonry-layout.js
class MasonryLayout {
    constructor(container, options = {}) {
        this.container = container;
        this.items = [];
        this.columns = options.columns || 10;
        this.gap = options.gap || 20;
        this.resizeTimer = null;
        this.isProcessing = false;
        this.queue = [];
        this.columnHeights = [];
        
        this.init();
    }
    
    init() {
        this.updateColumns();
        this.bindEvents();
        this.resetColumnHeights();
    }
    
    bindEvents() {
        window.addEventListener('resize', () => {
            clearTimeout(this.resizeTimer);
            this.resizeTimer = setTimeout(() => {
                this.updateColumns();
                this.relayout();
            }, 200);
        });
    }
    
    updateColumns() {
        const width = window.innerWidth;
        
        if (width <= 480) this.columns = 1;
        else if (width <= 768) this.columns = 2;
        else if (width <= 1024) this.columns = 3;
        else if (width <= 1280) this.columns = 4;
        else if (width <= 1440) this.columns = 5;
        else if (width <= 1600) this.columns = 6;
        else if (width <= 1920) this.columns = 7;
        else if (width <= 2200) this.columns = 8;
        else if (width <= 2400) this.columns = 9;
        else this.columns = 10;
        
        this.resetColumnHeights();
    }
    
    resetColumnHeights() {
        this.columnHeights = new Array(this.columns).fill(0);
    }
    
    clear() {
        // Limpa completamente o container e reseta alturas
        this.container.innerHTML = '';
        this.container.style.height = '0px';
        this.resetColumnHeights();
        this.items = [];
        console.log('Masonry limpo - pronto para novos items');
    }
    
    async addItem(element) {
        // Se for mobile (1 coluna), não usa posicionamento absoluto
        if (this.columns === 1) {
            element.style.position = 'relative';
            element.style.width = '100%';
            element.style.left = 'auto';
            element.style.top = 'auto';
            element.classList.add('show');
            return;
        }
        
        // Posição inicial invisível para cálculo
        element.style.opacity = '0';
        element.style.visibility = 'hidden';
        
        // Aguarda imagens carregarem
        await this.waitForImages(element);
        
        // Encontra a coluna mais curta
        let shortestColumn = 0;
        let minHeight = this.columnHeights[0];
        
        for (let i = 1; i < this.columns; i++) {
            if (this.columnHeights[i] < minHeight) {
                minHeight = this.columnHeights[i];
                shortestColumn = i;
            }
        }
        
        // Calcula largura responsiva baseada no número de colunas atual
        const containerWidth = this.container.offsetWidth;
        const itemWidth = (containerWidth - (this.columns - 1) * this.gap) / this.columns;
        
        // Define posição final
        const finalLeft = shortestColumn * (itemWidth + this.gap);
        const finalTop = this.columnHeights[shortestColumn];
        
        element.style.position = 'absolute';
        element.style.left = finalLeft + 'px';
        element.style.top = (finalTop + 10) + 'px'; // Inicia 10px abaixo (menor deslocamento)
        element.style.width = itemWidth + 'px'; // Define largura responsiva
        element.style.visibility = 'visible';
        element.style.transform = 'translateY(10px)';
        element.style.transition = 'all 0.2s cubic-bezier(0.4, 0, 0.2, 1)';
        
        // Força reflow
        element.offsetHeight;
        
        // Anima para posição final
        requestAnimationFrame(() => {
            element.style.top = finalTop + 'px';
            element.style.opacity = '1';
            element.style.transform = 'translateY(0)';
            element.classList.add('show');
        });
        
        // Atualiza altura da coluna
        this.columnHeights[shortestColumn] += element.offsetHeight + this.gap;
        
        // Atualiza altura do container suavemente
        const maxHeight = Math.max(...this.columnHeights);
        this.container.style.transition = 'height 0.2s cubic-bezier(0.4, 0, 0.2, 1)';
        this.container.style.height = maxHeight + 'px';
        
        // Remove transição após animação
        setTimeout(() => {
            element.style.transition = '';
            this.container.style.transition = '';
        }, 200);
    }
    
    async waitForImages(element) {
        const images = element.querySelectorAll('img');
        const promises = [];
        
        images.forEach(img => {
            if (!img.complete) {
                promises.push(
                    new Promise(resolve => {
                        img.onload = resolve;
                        img.onerror = resolve;
                        setTimeout(resolve, 1000); // timeout mais rápido
                    })
                );
            }
        });
        
        if (promises.length > 0) {
            await Promise.all(promises);
        }
    }
    
    async relayout() {
        console.log('Relayout iniciado...');
        
        const elements = Array.from(this.container.querySelectorAll('.result'));
        
        if (elements.length === 0) return;
        
        // Se for mobile, usa layout simples
        if (this.columns === 1) {
            elements.forEach(element => {
                element.style.position = 'relative';
                element.style.width = '100%';
                element.style.left = 'auto';
                element.style.top = 'auto';
                element.classList.add('show');
            });
            this.container.style.height = 'auto';
            return;
        }
        
        // Reset das alturas das colunas
        this.resetColumnHeights();
        
        // Calcula dimensões responsivas baseadas no número de colunas atual
        const containerWidth = this.container.offsetWidth;
        const itemWidth = (containerWidth - (this.columns - 1) * this.gap) / this.columns;
        
        // Aguarda todas as imagens
        for (const element of elements) {
            await this.waitForImages(element);
        }
        
        // Reorganiza elementos
        for (let i = 0; i < elements.length; i++) {
            const element = elements[i];
            
            // Encontra coluna mais curta
            let shortestColumn = 0;
            let minHeight = this.columnHeights[0];
            
            for (let j = 1; j < this.columns; j++) {
                if (this.columnHeights[j] < minHeight) {
                    minHeight = this.columnHeights[j];
                    shortestColumn = j;
                }
            }
            
            // Calcula posição
            const left = shortestColumn * (itemWidth + this.gap);
            const top = this.columnHeights[shortestColumn];
            
            // Aplica estilos
            element.style.position = 'absolute';
            element.style.width = itemWidth + 'px'; // Define largura responsiva
            element.style.left = left + 'px';
            element.style.top = top + 'px';
            element.classList.add('show');
            
            // Atualiza altura da coluna
            this.columnHeights[shortestColumn] += element.offsetHeight + this.gap;
        }
        
        // Atualiza altura do container
        const maxHeight = Math.max(...this.columnHeights);
        this.container.style.height = maxHeight + 'px';
        
        console.log('Relayout concluído!');
    }
}

// Integração com o sistema existente
window.MasonryLayout = MasonryLayout;

// Aguarda o DOM estar pronto
document.addEventListener('DOMContentLoaded', function() {
    console.log('Masonry carregado, aguardando container...');
});

// Função global para debug
window.debugMasonry = function() {
    if (window.masonry) {
        console.log('Masonry Status:', {
            columns: window.masonry.columns,
            container: window.masonry.container,
            columnHeights: window.masonry.columnHeights,
            items: window.masonry.container.querySelectorAll('.result').length
        });
    } else {
        console.log('Masonry não inicializado');
    }
};

window.forceMasonryRelayout = function() {
    if (window.masonry) {
        window.masonry.updateColumns();
        window.masonry.relayout();
    }
};

window.clearMasonry = function() {
    if (window.masonry) {
        window.masonry.clear();
    }
};