/**
 * =========================================================
 *  File: patch.js
 *  Purpose: Non-invasive DOM enhancements on top of app JS
 *  Version: 2025-09-01
 * =========================================================
 */

(function(){
  // Utils
  function q(sel, root=document){ return root.querySelector(sel); }
  function qa(sel, root=document){ return Array.from(root.querySelectorAll(sel)); }
  function buildEl(html){ const t=document.createElement('template'); t.innerHTML = html.trim(); return t.content.firstElementChild; }
  function truncate(t, n){ return (t||'').length>n ? (t.slice(0,n-1)+'…') : (t||''); }

  function getFeedIdFromCard(card){
    // tenta por data-feed-id nos botões
    const btn = card.querySelector('.save-btn, .like-btn, .dislike-btn, .copy-embed-btn');
    if (btn && btn.dataset.feedId) return parseInt(btn.dataset.feedId,10);
    // fallback: tenta extrair de link para pindetails.php?id=123
    const a = card.querySelector('a[href*="pindetails.php?id="]');
    if (a){
      const m = a.href.match(/[?&]id=(\d+)/);
      if (m) return parseInt(m[1],10);
    }
    return null;
  }

  // ---------- SHARE sem EMBED ----------
  function buildShare(feed){
    const shareUrl = encodeURIComponent(feed.url || (`https://myhashtag.io/pindetails.php?id=${feed.id}`));
    const text = encodeURIComponent(feed.title || 'Check this out');
    const id = `share-${feed.id}`;
    const wrap = buildEl(`
      <div class="share-button-container">
        <button class="share-trigger" data-share-id="${id}" aria-expanded="false" title="Share">
          <i class="fas fa-share-alt"></i><span>Share</span>
        </button>
        <div class="share-options" id="${id}" role="menu" aria-hidden="true">
          <button data-href="https://www.facebook.com/sharer/sharer.php?u=${shareUrl}" title="Facebook"><i class="fab fa-facebook-f"></i></button>
          <button data-href="https://twitter.com/intent/tweet?url=${shareUrl}&text=${text}" title="X/Twitter"><i class="fab fa-twitter"></i></button>
          <button data-href="https://www.linkedin.com/sharing/share-offsite/?url=${shareUrl}" title="LinkedIn"><i class="fab fa-linkedin-in"></i></button>
          <button data-href="https://api.whatsapp.com/send?text=${text}%20${shareUrl}" title="WhatsApp"><i class="fab fa-whatsapp"></i></button>
          <button data-href="https://t.me/share/url?url=${shareUrl}&text=${text}" title="Telegram"><i class="fab fa-telegram-plane"></i></button>
          <button data-href="https://www.reddit.com/submit?url=${shareUrl}&title=${text}" title="Reddit"><i class="fab fa-reddit-alien"></i></button>
          <button class="copy-link" data-copy="${decodeURIComponent(shareUrl)}" title="Copy link"><i class="fas fa-link"></i></button>
        </div>
      </div>`);

    // toggle
    wrap.querySelector('.share-trigger').addEventListener('click', function(){
      const pid = this.dataset.shareId;
      const panel = document.getElementById(pid);
      const open = panel.classList.contains('open');
      document.querySelectorAll('.share-options.open').forEach(p => p.classList.remove('open'));
      document.querySelectorAll('.share-trigger[aria-expanded="true"]').forEach(b => b.setAttribute('aria-expanded','false'));
      panel.classList.toggle('open', !open);
      this.setAttribute('aria-expanded', (!open).toString());
      panel.setAttribute('aria-hidden', open.toString());
    });
    // clicks
    wrap.querySelectorAll('.share-options button').forEach(b=>{
      b.addEventListener('click', function(){
        const href = this.getAttribute('data-href');
        const copy = this.getAttribute('data-copy');
        if (copy) {
          if (navigator.clipboard?.writeText) navigator.clipboard.writeText(copy);
          else {
            const ta=document.createElement('textarea'); ta.value=copy; document.body.appendChild(ta); ta.select();
            try{ document.execCommand('copy'); }catch(e){}
            document.body.removeChild(ta);
          }
          return;
        }
        if (href) window.open(href, '_blank','noopener');
      });
    });

    return wrap;
  }

  // ---------- Comentários ----------
  function buildComments(feedId){
    const el = buildEl(`
      <div>
        <div class="comment-toggle">
          <button class="share-trigger open-comments" data-feed-id="${feedId}" aria-expanded="false" title="Open comments">
            <i class="far fa-comments"></i><span>Comments</span>
          </button>
        </div>
        <section class="comment-section" data-feed-id="${feedId}">
          <textarea placeholder="Write a comment…" maxlength="2000"></textarea>
          <button class="post-comment-btn" data-feed-id="${feedId}"><i class="fas fa-paper-plane"></i>&nbsp;Post</button>
          <div class="comments-list"></div>
        </section>
      </div>`);

    const section = el.querySelector('.comment-section');
    el.querySelector('.open-comments').addEventListener('click', function(){
      const open = section.style.display === 'block';
      section.style.display = open ? 'none' : 'block';
      if (!open) loadComments(feedId, section);
    });
    el.querySelector('.post-comment-btn').addEventListener('click', function(){
      const text = section.querySelector('textarea').value.trim();
      if (!text) return;
      postComment(feedId, text, section);
    });

    return el;
  }

  function loadComments(feedId, section){
    fetch(`api_comments.php?feed_id=${feedId}`)
      .then(r=>r.json()).then(res=>{
        if(!res?.success) return;
        const list = section.querySelector('.comments-list');
        list.innerHTML = '';
        (res.comments||[]).forEach(c=>{
          const when = new Date(c.created_at).toLocaleString();
          const pic = c.profile_picture ? `<img src="${c.profile_picture}" alt="@${c.username}" class="author-avatar" style="width:26px;height:26px;border-radius:50%;">` : `<i class="fas fa-user-circle" style="font-size:22px;margin:2px 6px 0 0;"></i>`;
          const node = buildEl(`
            <div class="comment">
              ${pic}
              <div>
                <div class="meta">@${c.username} · ${when}</div>
                <div class="text">${(c.comment_text||'').replace(/</g,'&lt;').replace(/>/g,'&gt;')}</div>
              </div>
            </div>`);
          list.appendChild(node);
        });
      });
  }
  function postComment(feedId, text, section){
    fetch('api_comments.php', {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify({ feed_id: feedId, comment_text: text })
    }).then(r=>r.json()).then(res=>{
      if(res?.success){
        section.querySelector('textarea').value = '';
        loadComments(feedId, section);
      }else if(res?.error==='not_logged_in'){
        window.location.href='login.html';
      }
    });
  }

  // ---------- Ações: garantir 1 linha ----------
  function compactIfOverflow(row){
    if (!row) return;
    // aguardar layout assentar
    requestAnimationFrame(()=>{
      const needsCompact = row.scrollWidth > row.clientWidth + 8;
      row.classList.toggle('compact', needsCompact);
    });
  }

  function ensureActionsRow(card){
    // Se já existe, apenas compacta
    let row = card.querySelector('.actions-row');
    const like = card.querySelector('.like-btn');
    const dislike = card.querySelector('.dislike-btn');
    const save = card.querySelector('.save-btn');
    let embed = card.querySelector('.copy-embed-btn');

    // se não existir embed copy, cria um
    if (!embed) {
      embed = buildEl(`<button class="action-btn copy-embed-btn" title="Copy embed code"><i class="fas fa-code"></i><span class="label">Embed</span></button>`);
      // tenta inferir feed id
      const id = getFeedIdFromCard(card);
      if (id) embed.dataset.feedId = id;
      embed.addEventListener('click', function(){
        // tenta usar função global create embed (se houver) — senão, copia apenas a URL
        const fid = this.dataset.feedId || getFeedIdFromCard(card);
        try{
          // Se seus scripts expõem gerar IFRAME, podemos chamar. Caso não, fallback:
          if (window.feedsById && window.feedsById[fid] && window.generateIframeCode){
            const code = window.generateIframeCode(window.feedsById[fid]);
            if (navigator.clipboard?.writeText) navigator.clipboard.writeText(code);
            else { const ta=document.createElement('textarea'); ta.value=code; document.body.appendChild(ta); ta.select(); try{document.execCommand('copy');}catch(e){} document.body.removeChild(ta); }
          }else{
            const a = card.querySelector('a[href*="pindetails.php?id="]');
            const link = a ? a.href : location.href;
            if (navigator.clipboard?.writeText) navigator.clipboard.writeText(link);
          }
        }catch(e){}
      });
    }

    if (!row) {
      row = buildEl(`<div class="actions-row"></div>`);
      // inserir row num ponto estável (antes do share)
      const shareBlock = card.querySelector('.share-button-container') || card.lastElementChild;
      card.insertBefore(row, shareBlock);
    }

    // Zera conteúdo e insere na ordem correta, evitando duplicatas
    row.innerHTML = '';
    [like, dislike, save, embed].forEach(b=>{ if (b) row.appendChild(b); });

    compactIfOverflow(row);
    // recompacta em resize
    window.addEventListener('resize', ()=>compactIfOverflow(row));
  }

  // ---------- Share: substituir container antigo ----------
  function replaceShare(card){
    const id = getFeedIdFromCard(card);
    // tenta obter url/título
    const linkEl = card.querySelector('a[href*="pindetails.php?id="]') || card.querySelector('a[href]');
    const titleEl = card.querySelector('h2');
    const feed = {
      id: id || 0,
      url: linkEl ? linkEl.href : location.href,
      title: titleEl ? titleEl.textContent.trim() : document.title
    };
    // remove qualquer share pré-existente do card
    qa('.share-button-container, .share-options, .a2a_kit', card).forEach(n=>n.remove());
    // cria e adiciona o share novo
    const anchor = card.querySelector('.comment-toggle') || card.lastElementChild;
    card.insertBefore(buildShare(feed), anchor);
  }

  // ---------- Comentários: anexar se ainda não existe ----------
  function ensureComments(card){
    if (card.querySelector('.comment-section')) return;
    const id = getFeedIdFromCard(card);
    if (!id) return;
    const block = buildComments(id);
    card.appendChild(block);
  }

  // ---------- Enhance Card ----------
  function enhanceCard(card){
    try{
      ensureActionsRow(card);
      replaceShare(card);
      ensureComments(card);
    }catch(e){ /* silencioso */ }
  }

  function enhanceAll(){
    qa('#feeds .result').forEach(enhanceCard);
  }

  // Observer para novos cards (infinite scroll)
  const feedsEl = document.getElementById('feeds');
  if (feedsEl) {
    const obs = new MutationObserver(muts=>{
      muts.forEach(m=>{
        m.addedNodes && m.addedNodes.forEach(node=>{
          if (node.nodeType===1){
            if (node.classList.contains('result')) enhanceCard(node);
            else qa('.result', node).forEach(enhanceCard);
          }
        });
      });
    });
    obs.observe(feedsEl, { childList:true, subtree:true });
  }

  // First pass
  if (document.readyState === 'loading'){
    document.addEventListener('DOMContentLoaded', enhanceAll);
  } else {
    enhanceAll();
  }
})();
