// emojiList.js - Sistema Completo de Emojis com Interface Moderna

// Categorias de emojis organizadas
const emojiCategories = {
    "😊 Rostos e Emoções": {
        icon: "😊",
        emojis: ['😀', '😃', '😄', '😁', '😆', '😅', '🤣', '😂', '🙂', '😉', '😊', '😇', '🥰', '😍', '🤩', '😘', '😗', '☺️', '😚', '😙', '🥲', '😏', '😌', '😔', '😪', '😴', '😷', '🤒', '🤕', '🤢', '🤮', '🤧', '🥵', '🥶', '🥴', '😵', '😵‍💫', '🤯', '🥱', '😕', '🫤', '😟', '🙁', '☹️', '😮', '😯', '😲', '😳', '🥺', '🥹', '😦', '😧', '😨', '😰', '😥', '😢', '😭', '😱', '😖', '😣', '😞', '😓', '😩', '😫', '😤', '😡', '😠', '🤬', '😈', '👿', '💀', '☠️']
    },
    
    "🐾 Animais e Natureza": {
        icon: "🐾",
        emojis: ['🐶', '🐱', '🐭', '🐹', '🐰', '🦊', '🐻', '🐼', '🐻‍❄️', '🐨', '🐯', '🦁', '🐮', '🐷', '🐸', '🐵', '🙈', '🙉', '🙊', '🐒', '🦍', '🦧', '🐔', '🐧', '🐦', '🐤', '🐣', '🐥', '🦆', '🦅', '🦉', '🦇', '🐺', '🐗', '🐴', '🦄', '🐝', '🐛', '🦋', '🐌', '🐞', '🐜', '🦟', '🦗', '🕷️', '🦂', '🐢', '🐍', '🦎', '🐙', '🦑', '🦐', '🦞', '🦀', '🐡', '🐠', '🐟', '🐬', '🐳', '🐋', '🦈', '🐊', '🐅', '🐆', '🦓', '🦒', '🦘', '🦬', '🐘', '🦛', '🦏', '🐪', '🐫', '🦙', '🐏', '🐑', '🐐', '🦌', '🐕', '🐩', '🦮', '🐈', '🐈‍⬛', '🐓', '🦃', '🦚', '🦜', '🦢', '🦩', '🕊️', '🐇', '🦝', '🦨', '🦡', '🦦', '🦥', '🐁', '🐀', '🦔']
    },
    
    "🌺 Plantas e Flores": {
        icon: "🌺",
        emojis: ['🌵', '🎄', '🌲', '🌳', '🌴', '🌱', '🌿', '☘️', '🍀', '🎍', '🎋', '🍃', '🍂', '🍁', '🍄', '🌾', '💐', '🌷', '🌹', '🥀', '🌺', '🌸', '🌼', '🌻', '🌞', '🌝', '🌛', '🌜', '🌚', '🌕', '🌖', '🌗', '🌘', '🌑', '🌒', '🌓', '🌔', '🌙', '🌎', '🌍', '🌏', '💫', '⭐', '🌟', '✨', '⚡', '☄️', '💥', '🔥', '🌪️', '🌈', '☀️', '🌤️', '⛅', '🌥️', '☁️', '🌦️', '🌧️', '⛈️', '🌩️', '🌨️', '❄️', '☃️', '⛄', '🌬️', '💨', '💧', '💦', '🌊', '🌫️']
    },
    
    "🍎 Comidas e Bebidas": {
        icon: "🍎",
        emojis: ['🍏', '🍎', '🍐', '🍊', '🍋', '🍌', '🍉', '🍇', '🍓', '🫐', '🍈', '🍒', '🍑', '🥭', '🍍', '🥥', '🥝', '🍅', '🍆', '🥑', '🥦', '🥬', '🥒', '🌶️', '🫑', '🌽', '🥕', '🫒', '🧄', '🧅', '🥔', '🍠', '🥐', '🥖', '🥨', '🧀', '🥚', '🍳', '🧈', '🥞', '🧇', '🥓', '🥩', '🍗', '🍖', '🌭', '🍔', '🍟', '🍕', '🫓', '🥪', '🥙', '🧆', '🌮', '🌯', '🫔', '🥗', '🥘', '🫕', '🥫', '🍝', '🍜', '🍲', '🍛', '🍣', '🍱', '🥟', '🦪', '🍤', '🍙', '🍚', '🍘', '🍥', '🥠', '🥮', '🍢', '🍡', '🍧', '🍨', '🍦', '🥧', '🧁', '🍰', '🎂', '🍮', '🍭', '🍬', '🍫', '🍿', '🍩', '🍪', '🌰', '🥜', '🍯', '🥛', '🍼', '🫖', '☕', '🍵', '🧃', '🥤', '🧋', '🍶', '🍺', '🍻', '🥂', '🍷', '🥃', '🍸', '🍹', '🧉', '🍾', '🧊', '🥄', '🍴', '🍽️', '🥣', '🥡', '🥢', '🧂']
    },
    
    "🏃 Atividades e Esportes": {
        icon: "🏃",
        emojis: ['⚽', '🏀', '🏈', '⚾', '🥎', '🏐', '🏉', '🎾', '🥏', '🎳', '🏏', '🏑', '🏒', '🥍', '🏓', '🏸', '🥊', '🥋', '🥅', '⛳', '⛸️', '🎣', '🤿', '🎽', '🎿', '🛷', '🥌', '🎯', '🪀', '🪁', '🎱', '🏆', '🏅', '🥇', '🥈', '🥉', '🏵️', '🎗️', '🎫', '🎟️', '🎪', '🤹', '🤸', '🤼', '🤾', '🤺', '🤳', '🏇', '🧘', '🏄', '🏊', '🚣', '🧗', '🚴', '🚵', '🏎️', '🏍️', '🎨', '🎬', '🎤', '🎧', '🎼', '🎹', '🥁', '🎷', '🎺', '🎸', '🪕', '🎻', '🎲', '♟️', '🎯', '🎳', '🎮', '🎰', '🧩']
    },
    
    "🚗 Transportes e Lugares": {
        icon: "🚗",
        emojis: ['🚗', '🚕', '🚙', '🚌', '🚎', '🏎️', '🚓', '🚑', '🚒', '🚐', '🛻', '🚚', '🚛', '🚜', '🏍️', '🛵', '🚲', '🛴', '🛹', '🛼', '🚁', '✈️', '🛫', '🛬', '🛩️', '🚀', '🛸', '🚤', '⛵', '🚢', '🛳️', '⛴️', '🛥️', '🚂', '🚆', '🚄', '🚅', '🚈', '🚇', '🚊', '🚉', '🚝', '🚞', '🚋', '🚃', '🚟', '🚠', '🚡', '🛤️', '⛽', '🚨', '🚥', '🚦', '🛑', '🚧', '⚓', '🏗️', '🏭', '🏠', '🏡', '🏘️', '🏚️', '🏢', '🏬', '🏣', '🏤', '🏥', '🏦', '🏨', '🏪', '🏫', '🏩', '💒', '🏛️', '⛪', '🕌', '🕍', '🛕', '🕋', '⛩️', '🛤️', '🗾', '🎡', '🎢', '🎠', '⛲', '⛱️', '🏖️', '🏝️', '🏜️', '🌋', '⛰️', '🏔️', '🗻', '🏕️', '⛺', '🏞️']
    },
    
    "💡 Objetos": {
        icon: "💡",
        emojis: ['⌚', '📱', '📲', '💻', '⌨️', '🖥️', '🖨️', '🖱️', '🖲️', '🕹️', '🗜️', '💽', '💾', '💿', '📀', '📼', '📷', '📸', '📹', '🎥', '📽️', '🎞️', '📞', '☎️', '📟', '📠', '📺', '📻', '🎙️', '🎚️', '🎛️', '🧭', '⏱️', '⏲️', '⏰', '🕰️', '⌛', '⏳', '📡', '🔋', '🔌', '💡', '🔦', '🕯️', '🪔', '🧯', '🛢️', '💸', '💵', '💴', '💶', '💷', '🪙', '💰', '💳', '💎', '⚖️', '🪜', '🧰', '🪛', '🔧', '🔨', '⚒️', '🛠️', '⛏️', '🪚', '🔩', '⚙️', '🪤', '🧱', '⛓️', '🧲', '🔫', '💣', '🧨', '🪓', '🔪', '🗡️', '⚔️', '🛡️', '🚬', '⚰️', '🪦', '⚱️', '🏺', '🔮', '📿', '🧿', '💈', '⚗️', '🔭', '🔬', '🕳️', '🩹', '🩺', '💊', '💉', '🩸', '🧬', '🦠', '🧫', '🧪', '🌡️', '🧹', '🪠', '🧺', '🧻', '🚽', '🚰', '🚿', '🛁', '🛀', '🧼', '🪥', '🪒', '🧽', '🪣', '🧴', '🛎️', '🔑', '🗝️', '🚪', '🪑', '🛋️', '🛏️', '🛌', '🧸', '🖼️', '🪞', '🪟', '🛍️', '🛒', '🎁', '🎈', '🎏', '🎀', '🪄', '🪅', '🎊', '🎉', '🎎', '🏮', '🎐', '🪩', '✉️', '📩', '📨', '📧', '💌', '📥', '📤', '📦', '🏷️', '🪧', '📪', '📫', '📬', '📭', '📮', '📯', '📜', '📃', '📄', '📑', '🧾', '📊', '📈', '📉', '🗒️', '🗓️', '📆', '📅', '🗑️', '📇', '🗃️', '🗳️', '🗄️', '📋', '📁', '📂', '🗂️', '🗞️', '📰', '📓', '📔', '📒', '📕', '📗', '📘', '📙', '📚', '📖', '🔖', '🧷', '🔗', '📎', '🖇️', '📐', '📏', '🧮', '📌', '📍', '✂️', '🖊️', '🖋️', '✒️', '🖌️', '🖍️', '📝', '✏️', '🔍', '🔎', '🔏', '🔐', '🔒', '🔓']
    },
    
    "⚡ Símbolos": {
        icon: "⚡",
        emojis: ['❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '🤎', '💔', '❤️‍🔥', '❤️‍🩹', '❣️', '💕', '💞', '💓', '💗', '💖', '💘', '💝', '💟', '☮️', '✝️', '☪️', '🕉️', '☸️', '✡️', '🔯', '🕎', '☯️', '☦️', '🛐', '⛎', '♈', '♉', '♊', '♋', '♌', '♍', '♎', '♏', '♐', '♑', '♒', '♓', '🆔', '⚛️', '🉑', '☢️', '☣️', '📴', '📳', '🈶', '🈚', '🈸', '🈺', '🈷️', '✴️', '🆚', '💮', '🉐', '㊙️', '㊗️', '🈴', '🈵', '🈹', '🈲', '🅰️', '🅱️', '🆎', '🆑', '🅾️', '🆘', '❌', '⭕', '🛑', '⛔', '📛', '🚫', '💯', '💢', '♨️', '🚷', '🚯', '🚳', '🚱', '🔞', '📵', '🚭', '❗', '❕', '❓', '❔', '‼️', '⁉️', '🔅', '🔆', '〽️', '⚠️', '🚸', '🔱', '⚜️', '🔰', '♻️', '✅', '🈯', '💹', '❇️', '✳️', '❎', '🌐', '💠', 'Ⓜ️', '🌀', '💤', '🏧', '🚾', '♿', '🅿️', '🛗', '🈳', '🈂️', '🛂', '🛃', '🛄', '🛅', '🚹', '🚺', '🚼', '⚧️', '🚻', '🚮', '🎦', '📶', '🈁', '🔣', 'ℹ️', '🔤', '🔡', '🔠', '🆖', '🆗', '🆙', '🆒', '🆕', '🆓', '0️⃣', '1️⃣', '2️⃣', '3️⃣', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟', '🔢', '#️⃣', '*️⃣', '⏏️', '▶️', '⏸️', '⏯️', '⏹️', '⏺️', '⏭️', '⏮️', '⏩', '⏪', '⏫', '⏬', '◀️', '🔼', '🔽', '➡️', '⬅️', '⬆️', '⬇️', '↗️', '↘️', '↙️', '↖️', '↕️', '↔️', '↪️', '↩️', '⤴️', '⤵️', '🔀', '🔁', '🔂', '🔄', '🔃', '🎵', '🎶', '➕', '➖', '➗', '✖️', '🟰', '♾️', '💲', '💱', '™️', '©️', '®️', '〰️', '➰', '➿', '🔚', '🔙', '🔛', '🔝', '🔜', '✔️', '☑️', '🔘', '⬜', '⬛', '◼️', '◻️', '🔳', '🔲', '▪️', '▫️', '◾', '◽', '🔸', '🔹', '🔶', '🔷', '🔺', '🔻', '💭', '🗨️', '💬', '🗯️', '♠️', '♣️', '♥️', '♦️', '🃏', '🎴', '🀄', '🕐', '🕑', '🕒', '🕓', '🕔', '🕕', '🕖', '🕗', '🕘', '🕙', '🕚', '🕛', '🕜', '🕝', '🕞', '🕟', '🕠', '🕡', '🕢', '🕣', '🕤', '🕥', '🕦', '🕧']
    },
    
    "👋 Gestos e Pessoas": {
        icon: "👋",
        emojis: ['👋', '🤚', '🖐️', '✋', '🖖', '🫱', '🫲', '🫳', '🫴', '👌', '🤌', '🤏', '✌️', '🤞', '🫰', '🤟', '🤘', '🤙', '👈', '👉', '👆', '🖕', '👇', '☝️', '🫵', '👍', '👎', '✊', '👊', '🤛', '🤜', '👏', '🙌', '🫶', '👐', '🤲', '🤝', '🙏', '✍️', '💅', '🤳', '💪', '🦾', '🦿', '🦵', '🦶', '👂', '🦻', '👃', '🧠', '🫀', '🫁', '🦷', '🦴', '👀', '👁️', '👅', '👄', '🫦', '👶', '🧒', '👦', '👧', '🧑', '👱', '👨', '🧔', '🧑‍🦰', '👨‍🦰', '👨‍🦱', '👨‍🦳', '👨‍🦲', '👩', '👩‍🦰', '🧑‍🦰', '👩‍🦱', '🧑‍🦱', '👩‍🦳', '🧑‍🦳', '👩‍🦲', '🧑‍🦲', '👱‍♀️', '👱‍♂️', '🧓', '👴', '👵', '🙍', '🙍‍♂️', '🙍‍♀️', '🙎', '🙎‍♂️', '🙎‍♀️', '🙅', '🙅‍♂️', '🙅‍♀️', '🙆', '🙆‍♂️', '🙆‍♀️', '💁', '💁‍♂️', '💁‍♀️', '🙋', '🙋‍♂️', '🙋‍♀️', '🧏', '🧏‍♂️', '🧏‍♀️', '🙇', '🙇‍♂️', '🙇‍♀️', '🤦', '🤦‍♂️', '🤦‍♀️', '🤷', '🤷‍♂️', '🤷‍♀️', '🧑‍⚕️', '👨‍⚕️', '👩‍⚕️', '🧑‍🎓', '👨‍🎓', '👩‍🎓', '🧑‍🏫', '👨‍🏫', '👩‍🏫', '🧑‍⚖️', '👨‍⚖️', '👩‍⚖️', '🧑‍🌾', '👨‍🌾', '👩‍🌾', '🧑‍🍳', '👨‍🍳', '👩‍🍳', '🧑‍🔧', '👨‍🔧', '👩‍🔧', '🧑‍🏭', '👨‍🏭', '👩‍🏭', '🧑‍💼', '👨‍💼', '👩‍💼', '🧑‍🔬', '👨‍🔬', '👩‍🔬', '🧑‍💻', '👨‍💻', '👩‍💻', '🧑‍🎤', '👨‍🎤', '👩‍🎤', '🧑‍🎨', '👨‍🎨', '👩‍🎨', '🧑‍✈️', '👨‍✈️', '👩‍✈️', '🧑‍🚀', '👨‍🚀', '👩‍🚀', '🧑‍🚒', '👨‍🚒', '👩‍🚒', '👮', '👮‍♂️', '👮‍♀️', '🕵️', '🕵️‍♂️', '🕵️‍♀️', '💂', '💂‍♂️', '💂‍♀️', '🥷', '👷', '👷‍♂️', '👷‍♀️', '🫅', '🤴', '👸', '👳', '👳‍♂️', '👳‍♀️', '👲', '🧕', '🤵', '🤵‍♂️', '🤵‍♀️', '👰', '👰‍♂️', '👰‍♀️', '🤰', '🫃', '🫄', '🤱', '👩‍🍼', '👨‍🍼', '🧑‍🍼', '👼', '🎅', '🤶', '🧑‍🎄', '🦸', '🦸‍♂️', '🦸‍♀️', '🦹', '🦹‍♂️', '🦹‍♀️', '🧙', '🧙‍♂️', '🧙‍♀️', '🧚', '🧚‍♂️', '🧚‍♀️', '🧛', '🧛‍♂️', '🧛‍♀️', '🧜', '🧜‍♂️', '🧜‍♀️', '🧝', '🧝‍♂️', '🧝‍♀️', '🧞', '🧞‍♂️', '🧞‍♀️', '🧟', '🧟‍♂️', '🧟‍♀️', '🧌', '💆', '💆‍♂️', '💆‍♀️', '💇', '💇‍♂️', '💇‍♀️', '🚶', '🚶‍♂️', '🚶‍♀️', '🧍', '🧍‍♂️', '🧍‍♀️', '🧎', '🧎‍♂️', '🧎‍♀️', '🧑‍🦯', '👨‍🦯', '👩‍🦯', '🧑‍🦼', '👨‍🦼', '👩‍🦼', '🧑‍🦽', '👨‍🦽', '👩‍🦽', '🏃', '🏃‍♂️', '🏃‍♀️', '💃', '🕺', '🕴️', '👯', '👯‍♂️', '👯‍♀️', '🧖', '🧖‍♂️', '🧖‍♀️', '🧘', '🧑‍🤝‍🧑', '👭', '👫', '👬', '💏', '👩‍❤️‍💋‍👨', '👨‍❤️‍💋‍👨', '👩‍❤️‍💋‍👩', '💑', '👩‍❤️‍👨', '👨‍❤️‍👨', '👩‍❤️‍👩', '👪', '👨‍👩‍👦', '👨‍👩‍👧', '👨‍👩‍👧‍👦', '👨‍👩‍👦‍👦', '👨‍👩‍👧‍👧', '👨‍👨‍👦', '👨‍👨‍👧', '👨‍👨‍👧‍👦', '👨‍👨‍👦‍👦', '👨‍👨‍👧‍👧', '👩‍👩‍👦', '👩‍👩‍👧', '👩‍👩‍👧‍👦', '👩‍👩‍👦‍👦', '👩‍👩‍👧‍👧', '👨‍👦', '👨‍👦‍👦', '👨‍👧', '👨‍👧‍👦', '👨‍👧‍👧', '👩‍👦', '👩‍👦‍👦', '👩‍👧', '👩‍👧‍👦', '👩‍👧‍👧', '🗣️', '👤', '👥', '🫂', '👣']
    },
    
    "🎉 Eventos e Celebrações": {
        icon: "🎉",
        emojis: ['🎃', '🎄', '🎆', '🎇', '🧨', '✨', '🎈', '🎉', '🎊', '🎋', '🎍', '🎎', '🎏', '🎐', '🎑', '🧧', '🎁', '🎗️', '🎟️', '🎫', '🎖️', '🏆', '🏅', '🥇', '🥈', '🥉', '⚽', '⚾', '🥎', '🏀', '🏐', '🏈', '🏉', '🎾', '🥏', '🎳', '🏏', '🏑', '🏒', '🥍', '🏓', '🏸', '🥊', '🥋', '🥅', '⛳', '⛸️', '🎣', '🤿', '🎽', '🎿', '🛷', '🥌']
    },
    
    "🏳️ Bandeiras": {
        icon: "🏳️",
        emojis: ['🏁', '🚩', '🎌', '🏴', '🏳️', '🏳️‍🌈', '🏳️‍⚧️', '🏴‍☠️', '🇦🇨', '🇦🇩', '🇦🇪', '🇦🇫', '🇦🇬', '🇦🇮', '🇦🇱', '🇦🇲', '🇦🇴', '🇦🇶', '🇦🇷', '🇦🇸', '🇦🇹', '🇦🇺', '🇦🇼', '🇦🇽', '🇦🇿', '🇧🇦', '🇧🇧', '🇧🇩', '🇧🇪', '🇧🇫', '🇧🇬', '🇧🇭', '🇧🇮', '🇧🇯', '🇧🇱', '🇧🇲', '🇧🇳', '🇧🇴', '🇧🇶', '🇧🇷', '🇧🇸', '🇧🇹', '🇧🇻', '🇧🇼', '🇧🇾', '🇧🇿', '🇨🇦', '🇨🇨', '🇨🇩', '🇨🇫', '🇨🇬', '🇨🇭', '🇨🇮', '🇨🇰', '🇨🇱', '🇨🇲', '🇨🇳', '🇨🇴', '🇨🇵', '🇨🇷', '🇨🇺', '🇨🇻', '🇨🇼', '🇨🇽', '🇨🇾', '🇨🇿', '🇩🇪', '🇩🇬', '🇩🇯', '🇩🇰', '🇩🇲', '🇩🇴', '🇩🇿', '🇪🇦', '🇪🇨', '🇪🇪', '🇪🇬', '🇪🇭', '🇪🇷', '🇪🇸', '🇪🇹', '🇪🇺', '🇫🇮', '🇫🇯', '🇫🇰', '🇫🇲', '🇫🇴', '🇫🇷', '🇬🇦', '🇬🇧', '🇬🇩', '🇬🇪', '🇬🇫', '🇬🇬', '🇬🇭', '🇬🇮', '🇬🇱', '🇬🇲', '🇬🇳', '🇬🇵', '🇬🇶', '🇬🇷', '🇬🇸', '🇬🇹', '🇬🇺', '🇬🇼', '🇬🇾', '🇭🇰', '🇭🇲', '🇭🇳', '🇭🇷', '🇭🇹', '🇭🇺', '🇮🇨', '🇮🇩', '🇮🇪', '🇮🇱', '🇮🇲', '🇮🇳', '🇮🇴', '🇮🇶', '🇮🇷', '🇮🇸', '🇮🇹', '🇯🇪', '🇯🇲', '🇯🇴', '🇯🇵', '🇰🇪', '🇰🇬', '🇰🇭', '🇰🇮', '🇰🇲', '🇰🇳', '🇰🇵', '🇰🇷', '🇰🇼', '🇰🇾', '🇰🇿', '🇱🇦', '🇱🇧', '🇱🇨', '🇱🇮', '🇱🇰', '🇱🇷', '🇱🇸', '🇱🇹', '🇱🇺', '🇱🇻', '🇱🇾', '🇲🇦', '🇲🇨', '🇲🇩', '🇲🇪', '🇲🇫', '🇲🇬', '🇲🇭', '🇲🇰', '🇲🇱', '🇲🇲', '🇲🇳', '🇲🇴', '🇲🇵', '🇲🇶', '🇲🇷', '🇲🇸', '🇲🇹', '🇲🇺', '🇲🇻', '🇲🇼', '🇲🇽', '🇲🇾', '🇲🇿', '🇳🇦', '🇳🇨', '🇳🇪', '🇳🇫', '🇳🇬', '🇳🇮', '🇳🇱', '🇳🇴', '🇳🇵', '🇳🇷', '🇳🇺', '🇳🇿', '🇴🇲', '🇵🇦', '🇵🇪', '🇵🇫', '🇵🇬', '🇵🇭', '🇵🇰', '🇵🇱', '🇵🇲', '🇵🇳', '🇵🇷', '🇵🇸', '🇵🇹', '🇵🇼', '🇵🇾', '🇶🇦', '🇷🇪', '🇷🇴', '🇷🇸', '🇷🇺', '🇷🇼', '🇸🇦', '🇸🇧', '🇸🇨', '🇸🇩', '🇸🇪', '🇸🇬', '🇸🇭', '🇸🇮', '🇸🇯', '🇸🇰', '🇸🇱', '🇸🇲', '🇸🇳', '🇸🇴', '🇸🇷', '🇸🇸', '🇸🇹', '🇸🇻', '🇸🇽', '🇸🇾', '🇸🇿', '🇹🇦', '🇹🇨', '🇹🇩', '🇹🇫', '🇹🇬', '🇹🇭', '🇹🇯', '🇹🇰', '🇹🇱', '🇹🇲', '🇹🇳', '🇹🇴', '🇹🇷', '🇹🇹', '🇹🇻', '🇹🇼', '🇹🇿', '🇺🇦', '🇺🇬', '🇺🇲', '🇺🇳', '🇺🇸', '🇺🇾', '🇺🇿', '🇻🇦', '🇻🇨', '🇻🇪', '🇻🇬', '🇻🇮', '🇻🇳', '🇻🇺', '🇼🇫', '🇼🇸', '🇽🇰', '🇾🇪', '🇾🇹', '🇿🇦', '🇿🇲', '🇿🇼', '🏴󐁧󐁢󐁥󐁮󐁧󐁿', '🏴󐁧󐁢󐁳󐁣󐁴󐁿', '🏴󐁧󐁢󐁷󐁬󐁳󐁿']
    }
};

// Emojis recentes (carregados do localStorage)
let recentEmojis = [];

// Função para carregar emojis recentes
function loadRecentEmojis() {
    try {
        const stored = localStorage.getItem('recentEmojis');
        if (stored) {
            recentEmojis = JSON.parse(stored);
        }
    } catch (e) {
        console.warn('Erro ao carregar emojis recentes');
    }
}

// Função para salvar emoji recente
function saveRecentEmoji(emoji) {
    // Remover emoji se já existir
    recentEmojis = recentEmojis.filter(e => e !== emoji);
    
    // Adicionar no início
    recentEmojis.unshift(emoji);
    
    // Limitar a 30 emojis recentes
    recentEmojis = recentEmojis.slice(0, 30);
    
    // Salvar no localStorage
    try {
        localStorage.setItem('recentEmojis', JSON.stringify(recentEmojis));
    } catch (e) {
        console.warn('Erro ao salvar emojis recentes');
    }
}

// Função para gerar o HTML da lista de emojis
function getEmojiListHtml(feedId) {
    loadRecentEmojis();
    
    let emojiHtml = `
        <div id="emoji-list-${feedId}" class="emoji-list">
            <button class="close-btn" onclick="closeEmojiList(${feedId})">
                <i class="fas fa-times"></i>
            </button>
            
            <div class="emoji-search-container">
                <input type="text" 
                       class="emoji-search" 
                       id="emoji-search-${feedId}"
                       placeholder="Buscar emoji..." 
                       onkeyup="searchEmojis(${feedId})">
                <i class="fas fa-search"></i>
            </div>
            
            <div class="emoji-content" id="emoji-content-${feedId}">
    `;
    
    // Adicionar emojis recentes se houver
    if (recentEmojis.length > 0) {
        emojiHtml += `
            <div class="emoji-category" data-category="recent">
                <div class="emoji-category-header">
                    <span class="emoji-category-icon">🕐</span>
                    <span class="emoji-category-title">Recentes</span>
                </div>
                <div class="emoji-grid">
        `;
        
        recentEmojis.forEach(emoji => {
            emojiHtml += `
                <span onclick="addEmoji('${feedId}', '${emoji}')" 
                      class="emoji-item" 
                      title="${getEmojiName(emoji)}">
                    ${emoji}
                </span>
            `;
        });
        
        emojiHtml += `
                </div>
            </div>
        `;
    }
    
    // Adicionar categorias de emojis
    for (const [category, data] of Object.entries(emojiCategories)) {
        const categoryId = category.toLowerCase().replace(/[^a-z0-9]/g, '-');
        
        emojiHtml += `
            <div class="emoji-category" data-category="${categoryId}">
                <div class="emoji-category-header">
                    <span class="emoji-category-icon">${data.icon}</span>
                    <span class="emoji-category-title">${category}</span>
                </div>
                <div class="emoji-grid">
        `;
        
        data.emojis.forEach(emoji => {
            emojiHtml += `
                <span onclick="addEmoji('${feedId}', '${emoji}')" 
                      class="emoji-item" 
                      title="${getEmojiName(emoji)}">
                    ${emoji}
                </span>
            `;
        });
        
        emojiHtml += `
                </div>
            </div>
        `;
    }
    
    emojiHtml += `
            </div>
            
            <div class="emoji-footer">
                <div class="emoji-skin-tones" id="emoji-skin-tones-${feedId}">
                    <span class="skin-tone" data-tone="default">👋</span>
                    <span class="skin-tone" data-tone="light">👋🏻</span>
                    <span class="skin-tone" data-tone="medium-light">👋🏼</span>
                    <span class="skin-tone" data-tone="medium">👋🏽</span>
                    <span class="skin-tone" data-tone="medium-dark">👋🏾</span>
                    <span class="skin-tone" data-tone="dark">👋🏿</span>
                </div>
            </div>
        </div>
    `;
    
    return emojiHtml;
}

// Função para obter nome do emoji (placeholder)
function getEmojiName(emoji) {
    // Aqui você pode implementar um mapeamento de nomes
    return emoji;
}

// Função para alternar a visibilidade da lista de emojis
function toggleEmojiList(feedId) {
    const emojiList = document.getElementById(`emoji-list-${feedId}`);
    
    if (!emojiList) return;
    
    if (emojiList.classList.contains('show')) {
        closeEmojiList(feedId);
    } else {
        // Fechar outras listas abertas
        document.querySelectorAll('.emoji-list.show').forEach(list => {
            const id = list.id.match(/\d+/)[0];
            if (id !== feedId.toString()) {
                closeEmojiList(id);
            }
        });
        
        emojiList.classList.add('show');
        
        // Focar no campo de busca
        setTimeout(() => {
            const searchInput = document.getElementById(`emoji-search-${feedId}`);
            if (searchInput) {
                searchInput.focus();
            }
        }, 300);
    }
}

// Função para adicionar emoji ao campo de comentário
function addEmoji(feedId, emoji) {
    const commentBox = document.getElementById(`new-comment-${feedId}`);
    if (!commentBox) return;
    
    // Obter posição do cursor
    const start = commentBox.selectionStart;
    const end = commentBox.selectionEnd;
    const text = commentBox.value;
    
    // Inserir emoji na posição do cursor
    commentBox.value = text.substring(0, start) + emoji + text.substring(end);
    
    // Reposicionar cursor após o emoji
    const newPos = start + emoji.length;
    commentBox.setSelectionRange(newPos, newPos);
    
    // Focar no campo
    commentBox.focus();
    
    // Salvar como emoji recente
    saveRecentEmoji(emoji);
    
    // Trigger input event para auto-resize
    commentBox.dispatchEvent(new Event('input', { bubbles: true }));
    
    // Não fechar a lista automaticamente
}

// Função para fechar o contêiner de emojis
function closeEmojiList(feedId) {
    const emojiList = document.getElementById(`emoji-list-${feedId}`);
    if (emojiList) {
        emojiList.classList.remove('show');
    }
}

// Função para buscar emojis
function searchEmojis(feedId) {
    const searchTerm = document.getElementById(`emoji-search-${feedId}`).value.toLowerCase();
    const emojiContent = document.getElementById(`emoji-content-${feedId}`);
    const categories = emojiContent.querySelectorAll('.emoji-category');
    
    if (searchTerm === '') {
        // Mostrar todas as categorias
        categories.forEach(category => {
            category.style.display = 'block';
            const items = category.querySelectorAll('.emoji-item');
            items.forEach(item => {
                item.style.display = 'inline-flex';
            });
        });
        return;
    }
    
    // Filtrar emojis
    categories.forEach(category => {
        const items = category.querySelectorAll('.emoji-item');
        let hasVisibleItems = false;
        
        items.forEach(item => {
            const emoji = item.textContent;
            const title = item.title.toLowerCase();
            
            if (emoji.includes(searchTerm) || title.includes(searchTerm)) {
                item.style.display = 'inline-flex';
                hasVisibleItems = true;
            } else {
                item.style.display = 'none';
            }
        });
        
        // Esconder categoria se não tiver itens visíveis
        category.style.display = hasVisibleItems ? 'block' : 'none';
    });
}

// Skin tone selector
$(document).on('click', '.skin-tone', function() {
    const tone = $(this).data('tone');
    $(this).siblings().removeClass('selected');
    $(this).addClass('selected');
    
    // Aqui você pode implementar a lógica para aplicar o tom de pele
    // aos emojis que suportam modificadores de tom de pele
});

// Fechar lista de emojis ao clicar fora
document.addEventListener('click', function(event) {
    if (!event.target.closest('.emoji-list') && 
        !event.target.matches('[onclick*="toggleEmojiList"]')) {
        document.querySelectorAll('.emoji-list.show').forEach(list => {
            list.classList.remove('show');
        });
    }
});

// Atalhos de teclado para emojis
document.addEventListener('keydown', function(e) {
    // ESC para fechar lista de emojis
    if (e.key === 'Escape') {
        document.querySelectorAll('.emoji-list.show').forEach(list => {
            list.classList.remove('show');
        });
    }
});

// CSS adicional para a lista de emojis (adicionar ao styles.css)
const emojiListCSS = `
<style>
.emoji-list {
    position: absolute;
    left: 0;
    right: 0;
    top: 100%;
    margin-top: 10px;
    background: rgba(30, 30, 30, 0.98);
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 20px;
    padding: 20px;
    width: 100%;
    max-height: 300px;
    overflow: hidden;
    box-shadow: 0 16px 48px rgba(0, 0, 0, 0.5);
    display: none;
    opacity: 0;
    transform: translateY(10px) scale(0.95);
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    z-index: 1000;
}

.emoji-list.show {
    display: block;
    opacity: 1;
    transform: translateY(0) scale(1);
}

.emoji-list .close-btn {
    position: absolute;
    top: 15px;
    right: 15px;
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    border-radius: 50%;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    color: #f0f0f0;
    font-size: 14px;
    transition: all 0.2s ease;
    z-index: 2;
}

.emoji-list .close-btn:hover {
    background: rgba(255, 255, 255, 0.2);
    color: #fff;
    transform: rotate(90deg);
}

.emoji-search-container {
    position: relative;
    margin-bottom: 15px;
}

.emoji-search {
    width: 100%;
    padding: 10px 40px 10px 15px;
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    color: #f0f0f0;
    font-size: 14px;
    outline: none;
    transition: all 0.2s ease;
}

.emoji-search:focus {
    background: rgba(255, 255, 255, 0.08);
    border-color: rgba(102, 126, 234, 0.5);
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.2);
}

.emoji-search-container i {
    position: absolute;
    right: 15px;
    top: 50%;
    transform: translateY(-50%);
    color: #808080;
    font-size: 14px;
}

.emoji-content {
    overflow-y: auto;
    max-height: 200px;
    margin-right: -10px;
    padding-right: 10px;
}

.emoji-content::-webkit-scrollbar {
    width: 6px;
}

.emoji-content::-webkit-scrollbar-track {
    background: rgba(255, 255, 255, 0.05);
    border-radius: 3px;
}

.emoji-content::-webkit-scrollbar-thumb {
    background: rgba(255, 255, 255, 0.2);
    border-radius: 3px;
}

.emoji-content::-webkit-scrollbar-thumb:hover {
    background: rgba(255, 255, 255, 0.3);
}

.emoji-category {
    margin-bottom: 20px;
}

.emoji-category-header {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 10px;
    padding-bottom: 8px;
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.emoji-category-icon {
    font-size: 20px;
}

.emoji-category-title {
    font-size: 13px;
    font-weight: 600;
    color: #b0b0b0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.emoji-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(40px, 1fr));
    gap: 4px;
}

.emoji-item {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    font-size: 24px;
    cursor: pointer;
    border-radius: 8px;
    transition: all 0.2s ease;
    position: relative;
}

.emoji-item:hover {
    background: rgba(102, 126, 234, 0.2);
    transform: scale(1.2);
    z-index: 1;
}

.emoji-item:active {
    transform: scale(1.1);
}

.emoji-footer {
    margin-top: 15px;
    padding-top: 15px;
    border-top: 1px solid rgba(255, 255, 255, 0.1);
}

.emoji-skin-tones {
    display: flex;
    justify-content: center;
    gap: 8px;
}

.skin-tone {
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    cursor: pointer;
    border-radius: 8px;
    border: 2px solid transparent;
    transition: all 0.2s ease;
}

.skin-tone:hover {
    background: rgba(255, 255, 255, 0.1);
    transform: scale(1.1);
}

.skin-tone.selected {
    border-color: #667eea;
    background: rgba(102, 126, 234, 0.2);
}

/* Mobile optimizations */
@media (max-width: 768px) {
    .emoji-list {
        position: fixed;
        bottom: 0;
        left: 0;
        right: 0;
        top: auto;
        margin: 0;
        border-radius: 20px 20px 0 0;
        max-height: 70vh;
        padding-bottom: env(safe-area-inset-bottom, 20px);
    }
    
    .emoji-grid {
        grid-template-columns: repeat(auto-fill, minmax(45px, 1fr));
    }
    
    .emoji-item {
        width: 45px;
        height: 45px;
        font-size: 28px;
    }
}
</style>
`;

// Adicionar CSS ao documento quando o DOM estiver pronto
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        const styleElement = document.createElement('style');
        styleElement.innerHTML = emojiListCSS.replace('<style>', '').replace('</style>', '');
        document.head.appendChild(styleElement);
    });
} else {
    const styleElement = document.createElement('style');
    styleElement.innerHTML = emojiListCSS.replace('<style>', '').replace('</style>', '');
    document.head.appendChild(styleElement);
}

// Exportar funções globais
window.getEmojiListHtml = getEmojiListHtml;
window.toggleEmojiList = toggleEmojiList;
window.addEmoji = addEmoji;
window.closeEmojiList = closeEmojiList;
window.searchEmojis = searchEmojis;